<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class VisitorMessageRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     * So the user must be guest
     * @return bool
     */
    public function authorize()
    {
        if (Auth::guest()) {
            return true;
        }
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule( 'visitorMessages' ) )
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules = [];
        if (Auth::user() && in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin'])){
            $admin_rules = [
                'visitor_email' => 'required|email',
                'reply' => 'required|min:10',
            ];

            $rules = array_merge($rules, $admin_rules);

        } else{

            $phonePattern = '/^(05){1}([0-9]{8})$/';


            $visitor_rules = [
                'name'     => ['required', 'min:3', 'max:100'],
                'email'     => ['required', 'email', 'min:3', 'max:100', 'unique:users,email'],
                'phone'=> [
                    "required",
                    "numeric",
                    "digits:10",
                    "regex:".$phonePattern,
                    'unique:users,phone'
                ],
                'subject'     => ['required', 'min:3', 'max:300'],
                'message'     => ['required', 'min:3', 'max:300'],
            ];

            $rules = array_merge($rules, $visitor_rules);
        }

        return $rules;
    }

    public function attributes()
    {
        $field = metaFields('visitorMessages');
        $local = getCurrentLocale();
        $attributes = [
            'name' => $field[$local]['name'] ?? __('visitorMessages.name'),
            'email' => $field[$local]['email'] ?? __('visitorMessages.email'),
            'phone' => $field[$local]['phone'] ?? __('visitorMessages.phone'),
            'subject' => $field[$local]['subject'] ?? __('visitorMessages.subject'),
            'message' => $field[$local]['message'] ?? __('visitorMessages.message'),
            'visitor_email' => $field[$local]['visitor_email'],
            'reply' => $field[$local]['reply'],
        ];

        return $attributes;
    }

    public function messages()
    {
        return [
            'phone.digits' => __('visitorMessages.phone_digits'),
        ];
    }

    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            $message = $validator->errors()->first();
            // dd($message);
            throw new HttpResponseException(response()->json(['message'=> $message, 'errors' => $errors], 400));
        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
