<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed send_to
 * @property mixed user_id
 * @property mixed instructor_id
 */
class UsersNotificationRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule( 'usersNotifications' ) )
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules = [
            'send_to' => 'required|in:all_user,all_provider',
            'user_id' => 'nullable',
            'title' => 'required|min:3|max:100',
            'message' => 'required|min:3|max:400',
        ];

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('usersNotifications');
        $local = getCurrentLocale();
        $attributes = [
            'title' => $field[$local]['title'] ?? __('usersNotifications.title'),
            'message' => $field[$local]['message'] ?? __('usersNotifications.message'),
            'send_to' => $field[$local]['send_to'] ?? __('usersNotifications.send_to'),
            'user_id'=> $field[$local]['user_id'] ?? __('usersNotifications.user_id'),
            ];

        return $attributes;
    }

    public function messages()
    {
        return [
            'user_id.required_if' => __('usersNotifications.user_id_required_if'),
        ];
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::ajax()){
            if ($validator->fails()){
                $errors = $validator->errors()->first();
                throw new HttpResponseException(response()->json(['errors' => $errors]));
            }

        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
