<?php

namespace App\Http\Requests;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed status
 */
class SettingRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('settings'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $taxNumberRegex = '/[0-9]+/';
        $rules=[
            'website_name' => 'required',
            'keywords' => 'required',
            'meta_description' => 'required',
            'site_email' => 'required|email',
            'site_address' => 'required',
            'site_phone' => 'required|required|regex:/^([\+00\-]?[0-9]{1,3}[0-9\s]{8,16})$/',
            'mail_driver' => 'required',
            'facebook' => 'required',
            'twitter' => 'required',
            'instagram' => 'required',
            'whatsapp' => 'required',
            'visa_master' => 'nullable|in:1,0',
            'apple_pay' => 'nullable|in:1,0',
            'stc_pay' => 'nullable|in:1,0',
            'mada' => 'nullable|in:1,0',
            'added_value' => 'required|numeric|min:1',
            'tax_number' => 'required|numeric|gt:0',
//            'website_percentage' => 'required|numeric|min:1',
//            'contact_us_image' => 'required',
            'mail_host' => 'required',
            'mail_port' => 'required|numeric',
            'mail_username' => 'required',
            'mail_password' => 'required',
            'mail_encryption' => 'nullable',
            'mail_from_address' => 'required',
            'mail_from_name' => 'required',
            'website_logo' => 'sometimes',
            'website_disable' => 'sometimes',
            'dashboard_disable' => 'sometimes',
        ];

//        foreach (languages() as $lang) {
//            $lang_rules=[
//                'offer_alert_'.$lang->local=> 'required|min:3',
//            ];
//            $rules= array_merge($rules, $lang_rules);
//        }

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('settings');
        $local = getCurrentLocale();

        $attributes =  [
            'website_name' => $field[$local]['website_name'] ?? __('settings.website_name'),
            'keywords' => $field[$local]['keywords'] ?? __('settings.keywords'),
            'meta_description' => $field[$local]['meta_description'] ?? __('settings.meta_description'),
            'site_email' => $field[$local]['site_email'] ?? __('settings.site_email'),
            'postbox' => $field[$local]['postbox'] ?? __('settings.postbox'),
            'site_address' => $field[$local]['site_address'] ?? __('settings.site_address'),
            'site_phone' => $field[$local]['site_phone'] ?? __('settings.site_phone'),
            'mail_driver' => $field[$local]['mail_driver'] ?? __('settings.mail_driver'),
            'mail_host' => $field[$local]['mail_host'] ?? __('settings.mail_host'),
            'facebook' => $field[$local]['facebook'] ?? __('settings.facebook'),
            'twitter' => $field[$local]['twitter'] ?? __('settings.twitter'),
            'instagram' => $field[$local]['instagram'] ?? __('settings.instagram'),
            'website_percentage' => $field[$local]['website_percentage'] ?? __('settings.website_percentage'),
            'visa_master' => $field[$local]['visa_master'] ?? __('settings.visa_master'),
            'apple_pay' => $field[$local]['apple_pay'] ?? __('settings.apple_pay'),
            'stc_pay' => $field[$local]['stc_pay'] ?? __('settings.stc_pay'),
            'mada' => $field[$local]['mada'] ?? __('settings.mada'),
            'contact_us_image' => $field[$local]['contact_us_image'] ?? __('settings.contact_us_image'),
            'mail_port' => $field[$local]['mail_port'] ?? __('settings.mail_port'),
            'mail_username' => $field[$local]['mail_username'] ?? __('settings.mail_username'),
            'mail_password' => $field[$local]['mail_password'] ?? __('settings.mail_password'),
            'mail_encryption' => $field[$local]['mail_encryption'] ?? __('settings.mail_encryption'),
            'mail_from_address' => $field[$local]['mail_from_address'] ?? __('settings.mail_from_address'),
            'mail_from_name' => $field[$local]['mail_from_name'] ?? __('settings.mail_from_name'),
            'website_logo' => $field[$local]['website_logo'] ?? __('settings.website_logo'),
            'tax_number' => $field[$local]['tax_number'] ?? __('settings.tax_number'),
        ];

        foreach (languages() as $lang) {
            $local = $lang->local == 'ar' ? '(عربي)' : '(English)';
            $lang_attributes=[
                'offer_alert_'.$lang->local=> ($field[$lang->local]['offer_alert'] ?? __('settings.offer_alert')) . $local,
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }

        return $attributes;
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
