<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class ServiceRequest extends FormRequest
{

    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('services'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules=[
//            'category_id' => 'required|exists:categories,id'
        'service_type' => 'required|in:person_count,days_count',
        'price' => 'numeric|min:1',
        ];
        foreach (languages() as $lang) {
            $lang_rules=[
                'name_'.$lang->local=> 'required|min:3|max:100',
            ];
            $rules= array_merge($rules, $lang_rules);
        }
        $rules['image'] = 'required';
        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {

        $field = metaFields('services');
        $local = getCurrentLocale();
        $attributes=[
//            'category_id' => $field[$local]['category_id'] ?? __('services.category_id')
            'service_type' => $field[$local]['service_type'] ?? __('services.service_type'),
            'image' => $field[$local]['image'] ?? __('services.image'),
            'price' => $field[$local]['price'] ?? __('services.price'),
        ];

        foreach (languages() as $lang) {
            $local = $lang->local == 'ar' ? '(عربي)' : '(English)';
            $lang_attributes=[
                'name_'.$lang->local=> ($field[$lang->local]['name'] ?? __('services.name')) . $local,
                'description_'.$lang->local=> ($field[$lang->local]['description'] ?? __('services.description')) . $local,
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }
        return $attributes;
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
    	// Change response attitude if the request done via Ajax requests like API requests

    	if(Request::wantsJson()){
	        $errors = (new ValidationException($validator))->errors();
	        throw new HttpResponseException(response()->json(['errors' => $errors], 422));
	    } else {
	    	// Do the original action of the Form request class
	    	parent::failedValidation($validator);
	    }
    }
}
