<?php

namespace App\Http\Requests;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed password
 * @property mixed active_way
 */
class ProviderUpdateProfileRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $phonePattern = '/^(05){1}([0-9]{8})$/';

        $rules=[
            'name'=> 'required|min:3',
//            'photo'=> 'nullable|mimes:png,jpg,jpeg|max:2048',
            'code' => 'required|exists:countries,code',
            'phone'=> [
                "required",
                "numeric",
//                "digits:10",
                "regex:".$phonePattern,
                Rule::unique('users')->ignore($this->user->id, 'id')
            ],
            'region_id'=> 'required|exists:regions,id',
            'city_id'=> 'required|exists:cities,id',
            'sector_id'=> 'required|exists:sectors,id',
            'street'=> 'required|min:3|max:200',
            'password'=> 'nullable|confirmed',
            'logo'=> 'nullable|mimes:png,jpg,jpeg|max:2048',
            'email_receive'=> 'nullable|in:1,0',
            'bank_name'=> 'required|min:3|max:50',
            'iban'=> 'required|min:3|max:50',
            'client_name'=> 'required|min:3|max:50',
            'acc_no'=> 'required|min:3|max:50',

        ];

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('members');
        $local = getCurrentLocale();
        $attributes= [
            'name'=> $field[$local]['name'] ?? __('members.name'),
            'phone'=> $field[$local]['phone'] ?? __('members.phone'),
            'password'=> $field[$local]['password'] ?? __('members.password'),
            'code'=> $field[$local]['code'] ?? __('members.code'),
            'region_id'=> $field[$local]['region_id'] ?? __('members.region_id'),
            'city_id'=> $field[$local]['city_id'] ?? __('members.city_id'),
            'sector_id'=> $field[$local]['sector_id'] ?? __('members.sector_id'),
            'street'=> $field[$local]['street'] ?? __('members.street'),
            'bank_name'=> $field[$local]['bank_name'] ?? __('members.bank_name'),
            'iban'=> $field[$local]['iban'] ?? __('members.iban'),
            'client_name'=> $field[$local]['client_name'] ?? __('members.client_name'),
            'acc_no'=> $field[$local]['acc_no'] ?? __('members.acc_no'),
            'email_receive'=> $field[$local]['email_receive'] ?? __('members.email_receive'),
            'password_confirmation'=> $field[$local]['password_confirmation'] ?? __('members.password_confirmation'),
        ];
        return $attributes;
    }

    public function messages()
    {
        return [
            'phone.digits' => __('members.phone_digits'),
            'phone.regex' => __('members.phone_regex'),
        ];
    }

    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::ajax()){
            if ($validator->fails()){
                $errors = $validator->errors()->first();
                throw new HttpResponseException(response()->json(['errors' => $errors]));
            }

        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
