<?php

namespace App\Http\Requests;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed email
 */

class NewsletterRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (Auth::guest()) {
            return true;
        }

        if (! Auth::check()) {
            return false;
        }

        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule( 'newsletters' ) )
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules = [];

        if (Auth::user() && in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']) ){
            $admin_rules = [
                'email' => 'required|email',
                'subject' => 'required|min:3',
                'message' => 'required|min:10',
                'categories.*' => 'sometimes',
            ];

            $rules = array_merge($rules, $admin_rules);

        } else{

            $visitor_rules = [
                'email' => 'required|email|unique:newsletters|max:200',
                ];

            $rules = array_merge($rules, $visitor_rules);
        }

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('newsletters');
        $local = getCurrentLocale();
        $attributes = [
            'email' => $field[$local]['email'] ?? __('newsletters.email'),
            'subject' => $field[$local]['subject'] ?? __('newsletters.subject'),
            'message' => $field[$local]['message'] ?? __('newsletters.message'),
        ];

        return $attributes;
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array
     */
    public function messages()
    {
        $mesages = [
            'email.unique' => __('newsletters.email_subscribed_before'),
        ];

        return $mesages;
    }

    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            $message = $validator->errors()->first();
            // dd($message);
            throw new HttpResponseException(response()->json(['message'=> $message, 'errors' => $errors], 400));
        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
