<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class HallRequest extends FormRequest
{

    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('halls'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules=[
        'user_id' => 'required|exists:users,id',
        'category_id' => 'required|exists:categories,id',
        'is_offer' => 'nullable|in:1,0',
        'hall_status' => 'nullable|in:1,0',
        'region_id' => 'required|exists:regions,id',
        'city_id' => 'required|exists:cities,id',
        'sector_id' => 'required|exists:sectors,id',
        'street' => 'required|min:3|max:200',
        'lat' => 'required|numeric|max:100',
        'lng' => 'required|numeric|max:100',
        'reservation_way' => 'required|in:direct,accept_request',
        'reservation_type' => 'required|in:person_count,days_count',
        'capacity' => 'required|numeric|min:1',
        'min_capacity' => 'required|numeric|min:1',
        'total_capacity' => 'required|numeric|min:1',
        'booking_amount_per_person' => 'required_if:reservation_type,person_count|nullable|gt:0',
        'booking_amount_per_day' => 'required_if:reservation_type,days_count|nullable|gt:0',
        'table_ids' => 'required|array|min:1|exists:tables,id',
        'occasions' => 'required|array|exists:occasions,id',
//        'added_services_name' => 'required|array',
//        'added_services_description' => 'required|array',
//        'added_services_price' => 'required|array',
//        'added_services_image' => 'required|array',
//        'images' => 'required|array',
        ];
        foreach (languages() as $lang) {
            $lang_rules=[
                'name_'.$lang->local=> 'required|min:3|max:100',
                'description_'.$lang->local=> 'required|min:3',
                'reservation_condition_'.$lang->local=> 'required|min:3',
                'features_'.$lang->local=> 'required|min:3',
            ];
            $rules= array_merge($rules, $lang_rules);
        }


        $rules['dates_rang_from'] = 'required|date|after_or_equal:'.date('Y-m-d');
        $rules['dates_rang_to'] = 'required|date|after_or_equal:dates_rang_from';

        if (request('unavailable_dates_from')){
            foreach (request('unavailable_dates_from') as $from) {
                if (!$from){
                    $rules['all_unavailable_dates_from'] = 'required';
                }
            }
        }

        if (request('unavailable_dates_to')){
            foreach (request('unavailable_dates_to') as $key => $to) {
                if (!$to){
                    $rules['all_unavailable_dates_to'] = 'required';
                }
                if (strtotime($to) < strtotime(request('unavailable_dates_from')[$key])){
                    $rules['all_unavailable_dates_to_gt_from'] = 'required|after:unavailable_dates_from';
                }
            }
        }

        if (request('discount_dates_from')){
            foreach (request('discount_dates_from') as $from) {
                if (!$from){
                    $rules['all_discount_dates_from'] = 'required';
                }
            }
        }

        if (request('discount_dates_to')){
            foreach (request('discount_dates_to') as $key => $to) {
                if (!$to){
                    $rules['all_discount_dates_to'] = 'required';
                }
                if (strtotime($to) < strtotime(request('discount_dates_from')[$key])){
                    $rules['all_discount_dates_to_gt_from'] = 'required|after:discount_dates_from';
                }
            }
        }

        if (request('discount_percentage')){
            foreach (request('discount_percentage') as $percentage) {
                if (!$percentage || $percentage <= 0){
                    $rules['all_discount_percentage'] = 'required|numeric|min:1|not_in:0|gt:0';
                }
            }
        }


        if (request('added_services_name')){
            $rules['added_services_name'] = 'required|array';
            foreach (request('added_services_name') as $name) {
                if (!$name){
                    $rules['all_added_services_name'] = 'required';
                }
            }
        }

        if (request('added_services_price')){
            $rules['added_services_price'] = 'required|array';
            foreach (request('added_services_price') as $price) {
                if (!$price){
                    $rules['all_added_services_price'] = 'required';
                }
            }
        }

        if (request('added_services_description')){
            $rules['added_services_description'] = 'required|array';
            foreach (request('added_services_description') as $desc) {
                if (!$desc){
                    $rules['all_added_services_description'] = 'required';
                }
            }
        }

        if (request('added_services_image')){
            $rules['added_services_image'] = 'required|array';
            foreach (request('added_services_image') as $image) {
                if (!$image){
                    $rules['all_added_services_image'] = 'required';
                }
            }
        }

        $rules['images'] = 'required|array';

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {

        $field = metaFields('halls');
        $local = getCurrentLocale();
        $attributes=[
            'user_id' => $field[$local]['user_id'] ?? __('halls.user_id'),
            'category_id' => $field[$local]['category_id'] ?? __('halls.category_id'),
            'table_ids' => $field[$local]['table_ids'] ?? __('halls.table_ids'),
            'occasions' => $field[$local]['occasions'] ?? __('halls.occasions'),
            'added_services_name' => $field[$local]['added_services_name'] ?? __('halls.added_services_name'),
            'added_services_description' => $field[$local]['added_services_description'] ?? __('halls.added_services_description'),
            'added_services_price' => $field[$local]['added_services_price'] ?? __('halls.added_services_price'),
            'added_services_image' => $field[$local]['added_services_image'] ?? __('halls.added_services_image'),
            'capacity' => $field[$local]['capacity'] ?? __('halls.capacity'),
            'total_capacity' => $field[$local]['total_capacity'] ?? __('halls.total_capacity'),
            'min_capacity' => $field[$local]['total_capacity'] ?? __('halls.min_capacity'),
            'reservation_way' => $field[$local]['reservation_way'] ?? __('halls.reservation_way'),
            'reservation_type' => $field[$local]['reservation_type'] ?? __('halls.reservation_type'),
            'region_id' => $field[$local]['region_id'] ?? __('halls.region_id'),
            'city_id' => $field[$local]['city_id'] ?? __('halls.city_id'),
            'sector_id' => $field[$local]['sector_id'] ?? __('halls.sector_id'),
            'street' => $field[$local]['street'] ?? __('halls.street'),
            'lat' => $field[$local]['lat'] ?? __('halls.lat'),
            'lng' => $field[$local]['lng'] ?? __('halls.lng'),
            'images' => $field[$local]['images'] ?? __('halls.images'),
            'booking_amount_per_person' => $field[$local]['booking_amount_per_person'] ?? __('halls.booking_amount_per_person'),
            'all_added_services_name' => $field[$local]['added_services_name'] ?? __('halls.added_services_name'),
            'all_available_dates_from' => $field[$local]['available_dates_from'] ?? __('halls.available_dates_from'),
            'all_available_dates_to' => $field[$local]['available_dates_to'] ?? __('halls.available_dates_to'),
            'all_added_services_price' => $field[$local]['added_services_price'] ?? __('halls.added_services_price'),
            'all_added_services_description' => $field[$local]['added_services_description'] ?? __('halls.added_services_description'),
            'all_added_services_image' => $field[$local]['added_services_image'] ?? __('halls.added_services_image'),
            'all_reserved_date' => $field[$local]['reserved_date'] ?? __('halls.reserved_date'),
            'all_discount_percentage' => $field[$local]['discount_percentage'] ?? __('halls.discount_percentage'),
            'is_offer' => $field[$local]['is_offer'] ?? __('halls.is_offer'),
            'hall_status' => $field[$local]['hall_status'] ?? __('halls.status'),
            'all_unavailable_dates_from' => $field[$local]['unavailable_dates_from'] ?? __('halls.unavailable_dates_from'),
            'all_unavailable_dates_to' =>  __('halls.unavailable_dates_to'),
            'all_unavailable_dates_to_gt_from' =>  __('halls.to_gt_from'),
            'all_discount_dates_from' => $field[$local]['discount_dates_from'] ?? __('halls.discount_dates_from'),
            'all_discount_dates_to' => $field[$local]['discount_dates_to'] ?? __('halls.discount_dates_to'),
            'dates_rang_from' => $field[$local]['dates_rang_from'] ?? __('halls.dates_rang_from'),
            'dates_rang_to' => $field[$local]['dates_rang_to'] ?? __('halls.dates_rang_to'),
        ];

        foreach (languages() as $lang) {
            $local = $lang->local == 'ar' ? '(عربي)' : '(English)';
            $lang_attributes=[
                'name_'.$lang->local=> ($field[$lang->local]['name'] ?? __('halls.name')) . $local,
                'description_'.$lang->local=> ($field[$lang->local]['description'] ?? __('halls.description')) . $local,
                'reservation_condition_'.$lang->local=> ($field[$lang->local]['reservation_condition'] ?? __('halls.reservation_condition')) . $local,
                'features_'.$lang->local=> ($field[$lang->local]['features'] ?? __('halls.features')) . $local,
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }
        return $attributes;
    }

    public function messages()
    {
        return [
            'all_discount_percentage.required' => __('halls.all_discount_percentage_required'),
            'booking_amount_per_person.required_if' => __('halls.booking_amount_per_person_required_if'),
            'booking_amount_per_person.gt' => __('halls.booking_amount_per_person_required_if'),
            'booking_amount_per_day.required_if' => __('halls.booking_amount_per_day_required_if'),
            'booking_amount_per_day.gt' => __('halls.booking_amount_per_day_required_if'),
            'all_added_services_price.gt' => __('halls.all_added_services_price_gt'),
            'all_added_services_price.numeric' => __('halls.all_added_services_price_gt'),
            'all_unavailable_dates_to_gt_from.required' => __('halls.unavailable_to_gt_from'),
            'all_discount_dates_to_gt_from.required' => __('halls.discount_to_gt_from'),

        ];
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
    	// Change response attitude if the request done via Ajax requests like API requests

    	if(Request::ajax()){
	        $errors = $validator->errors()->first();
	        throw new HttpResponseException(response()->json(['errors' => $errors]));
	    } else {
	    	// Do the original action of the Form request class
	    	parent::failedValidation($validator);
	    }
    }
}
