<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class AboutRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule( 'abouts' ) )
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules=[
            'image' => 'required',
            'video' => 'required|url',
            ];

        foreach (languages() as $lang) {
            $lang_rules=[
                'title_'.$lang->local=> 'required|min:3|max:100',
                'description_'.$lang->local=> 'required|min:3',
                'about_us_list_'.$lang->local=> 'required',
                'why_us_first_title_'.$lang->local=> 'required|min:3|max:100',
                'why_us_second_title_'.$lang->local=> 'required|min:3|max:1003',
                'why_us_third_title_'.$lang->local=> 'required|min:3|max:100',
                'why_us_first_description_'.$lang->local=> 'required|min:3',
                'why_us_second_description_'.$lang->local=> 'required|min:3',
                'why_us_third_description_'.$lang->local=> 'required|min:3',
            ];
            $rules= array_merge($rules, $lang_rules);
        }

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('abouts');
        $local = getCurrentLocale();
        $attributes=[
            'image' => $field[$local]['image'] ?? __('abouts.image'),
            'video' => $field[$local]['video'] ?? __('abouts.video'),
        ];
        foreach (languages() as $lang) {
            $local = $lang->local == 'ar' ? '(عربي)' : '(English)';
            $lang_attributes=[
                'title_'.$lang->local=> ($field[$lang->local]['title'] ?? __('abouts.title')) . $local,
                'description_'.$lang->local=> ( $field[$lang->local]['description'] ?? __('abouts.description')) . $local,
                'about_us_list_'.$lang->local=> ($field[$lang->local]['about_us_list'] ?? __('abouts.about_us_list')) . $local,
                'why_us_first_title_'.$lang->local=> ($field[$lang->local]['why_us_first_title'] ?? __('abouts.why_us_first_title')) . $local,
                'why_us_second_title_'.$lang->local=> ($field[$lang->local]['why_us_second_title'] ?? __('abouts.why_us_second_title')) . $local,
                'why_us_third_title_'.$lang->local=> ($field[$lang->local]['why_us_third_title'] ?? __('abouts.why_us_third_title')) . $local,
                'why_us_first_description_'.$lang->local=> ($field[$lang->local]['why_us_first_description'] ?? __('abouts.why_us_first_description')) . $local,
                'why_us_second_description_'.$lang->local=> ($field[$lang->local]['why_us_second_description'] ?? __('abouts.why_us_second_description')) . $local,
                'why_us_third_description_'.$lang->local=> ($field[$lang->local]['why_us_third_description'] ?? __('abouts.why_us_third_description')) . $local,
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }
        return $attributes;
    }

    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
