<?php

namespace App\Http\Controllers\Web;

use App\Country;
use App\Evaluate;
use App\Hall;
use App\Helpers\ImageHelper;
use App\Helpers\Sms;
use App\Http\Requests\CheckPhoneRequest;
use App\Http\Requests\EvaluateRequest;
use App\Http\Requests\UserRegisterRequest;
use App\Http\Requests\UserUpdateProfileRequest;
use App\Region;
use App\Reservation;
use App\User;
use App\UsersNotification;
use Illuminate\Contracts\View\Factory;
use Illuminate\Foundation\Auth\ThrottlesLogins;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\File;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Lang;
use Illuminate\Validation\ValidationException;
use Illuminate\View\View;

class UserController extends Controller
{

    use ThrottlesLogins;

    /**
     * The maximum number of attempts to allow.
     *
     * @return int
     */
    protected $maxAttempts = 3;


    /**
     * The number of minutes to throttle for.
     *
     * @return int
     */
    protected $decayMinutes = 1;


    /**
     * The field name throttle trait will use for.
     *
     * @return string
     */
    public function username() : string
    {
        return 'phone';
    }

    protected $view = 'web.members.users.';
    protected $passwordView = 'web.members.password.';

    /**
     * User profile
     * @return Factory|View
     * @throws \Exception
     */
    public function index()
    {
        $user = Auth::user();
        if (!in_array($user->role, ['user'])){
            return  abort(404);
//            return redirect()->route('home');
        }

        $country = Country::withDescription($user->country_id);
        $state = Region::withDescription(null, $user->state_id);
        $countries = Country::withDescription();
        $states = Region::withDescription($user->country_id);
        $title = __('members.my_profile');

        $reservations = Reservation::userReservations(null, null, null, null, $user->id)->groupBy('re_status');
        $dailyReservations = Reservation::userReservations('daily');

        $notifications = UsersNotification::orderByDesc('id')
            ->where('send_to', '!=', 'all_provider')
            ->orWhere('user_id', $user->id)
            ->orWhere('send_to', 'all_user')
            ->cursor();

        $allNotifications = [];
        $notificationsCount = [];
        foreach ($notifications as $notification) {
            if (is_array($notification->is_read) && in_array($user->id, $notification->is_read)){
                continue;
            }
            if ((is_array($notification->is_read) && !in_array($user->id, $notification->is_read)) || !is_array($notification->is_read)){
                $allNotifications[] = $notification;
            }
        }

        $notificationsCount = count($allNotifications);

        return view($this->view.'profile', get_defined_vars());
    }

    public function hallEvaluate(EvaluateRequest $request)
    {

        $user = Auth::user();
        $hall = Hall::find($request->hall_id);
        $reservation = Reservation::find($request->reservation_id);
        $evaluate = Evaluate::where('hall_id', $hall->id)->where('user_id', $user->id)->first();

        if (!$hall){
            return response()->json(['errors' => __('halls.hall_not_found')]);
        }
        if (!$reservation){
            return response()->json(['errors' => __('halls.reservation_not_found')]);
        }
        if ($hall->id != $reservation->hall_id || $reservation->user_id != $user->id){
            return response()->json(['errors' => __('halls.evaluate_not_allowed')]);
        }
        if ($reservation->re_status != 'finished' || $evaluate){
            return response()->json(['errors' => __('halls.evaluate_not_allowed')]);
        }

        $data = $request->all();

        $data['cleaning'] = $request->cleaning ?? 0 ;
        $data['services'] = $request->services ?? 0 ;
        $data['staffs'] = $request->staffs ?? 0 ;
        $data['comforts'] = $request->comforts ?? 0 ;
        $data['financial_value'] = $request->financial_value ?? 0 ;
        $data['location'] = $request->location ?? 0 ;


        $evaluate_average = ($data['cleaning']+$data['services']+$data['staffs']+$data['comforts']+$data['financial_value']+$data['location'])/6;

        date_default_timezone_set('Asia/Riyadh');
//        date_default_timezone_set('Africa/Cairo');

        Evaluate::create([
            'comment' => $request->comment,
            'evaluate_average' => round($evaluate_average),
            'evaluate_details' => [
                'cleaning' => $data['cleaning'],
                'services' => $data['services'],
                'staffs' => $data['staffs'],
                'comforts' => $data['comforts'],
                'financial_value' => $data['financial_value'],
                'location' => $data['location'],
            ],
            'user_id' => $user->id,
            'hall_id' => $hall->id,
        ]);

        $reservation->update(['is_evaluate' => 1]);
        $evaluates = Evaluate::where('hall_id', $hall->id);
        $rate = $evaluates->sum('evaluate_average')/$evaluates->count();
        $hall->update(['rate' => $rate]);

        $provider = $hall->provider;
        $hallName = descriptions($hall)->name;

        $title = __('halls.hall_evaluate') . "($hallName)";
        $message = __('halls.hall_evaluate_message');
        $message = str_replace_array('@@', [$provider->name, $hallName, $user->name, round($evaluate_average)], $message);

        $userNotification = UsersNotification::create([
            'title' => $title,
            'message' => $message,
            'user_id' => $provider->id,
            'notification_type' => 'success',
        ]);

        $icon = asset('assets/web/images/icons/profile/noti-success.svg');
        $now = \Carbon\Carbon::now();
        $notificationTime = \Illuminate\Support\Carbon::make($reservation->created_at);
        $notificationTime = $now->diffInMinutes($notificationTime);
        $data = [
            'hall_evaluate' => 'yes',
            'notification_time' => $notificationTime . __('halls.minutes') . __('halls.ago'),
            'notification_date' => date('M Y d', strtotime($reservation->created_at)),
            'notification_icon' => $icon,
            'notification_class' => '',
            'reservation_container_id' => 'pills-completed',
            'badge' => 'badge_completed',
            'badge_title' => __('reservations.confirmed'),
            'notification_title' => $title,
            'notification_message' => $message,
            'notification_id' => $userNotification->id
        ];

        if (is_array($provider->device_token) && isset($provider->device_token['token'])){
            $notificationForProvider = new SendNotification($title, $message, $icon, $data, route('provider.profile'));
            $provider->notify($notificationForProvider);
        }
        return response()->json(['message' => __('halls.evaluate_successfully')]);

    }

    public function refreshNotification(Request $request)
    {
        if ($request->ajax()){
            $user = Auth::user();
            $notifications = UsersNotification::orderBy('id', 'DESC')
                    ->orWhere('user_id', $user->id)
                    ->orWhereIn('send_to', ['all_users'])
                    ->cursor() ?? [];
            $notificationsCount = $notifications->count();
            return view($this->view.'refresh_notification', compact('notifications', 'notificationsCount', 'user'));
        }
    }

    /** Its fired when user click on notification tab*/
    public function markAsRead(Request $request, $notification = null)
    {
        $user = Auth::user();

        if($request->ajax()){

            if ($request->mark_all_as_read){
                $notifications = UsersNotification::orWhere('user_id', $user->id)
                    ->orWhere('send_to', 'all_user')->get();

                foreach ($notifications as $notification) {
                    $isRead = is_array($notification->is_read) && count($notification->is_read) ? $notification->is_read : [];
                    array_push($isRead, $user->id);
                    $notification->update(['is_read' => $isRead]);
                }
            } else {
                $notification = UsersNotification::find($notification);
                if (!$notification){
                    return ['errors' => __('members.notification_not_found')];
                }
                $isRead = is_array($notification->is_read) && count($notification->is_read) ? $notification->is_read : [];
                if ($notification->user_id != $user->id){
                    return ['errors' => __('members.you_cant_delete_notification')];
                }
                if (!$notification->user_id && $notification->send_to != 'all_user'){
                    return ['errors' => __('members.you_cant_delete_notification')];
                }
                if ($notification->user_id == $user->id && in_array($user->id, $isRead)){
                    return ['errors' => __('members.notification_not_found')];
                }

                array_push($isRead, $user->id);
                $notification->update(['is_read' => $isRead]);
            }

            return response()->json(['success' => true]);
        }
    }

    public function updateProfile(UserUpdateProfileRequest $request, User $user)
    {
        $data = $request->all();
        $data['password'] = $request->password ? bcrypt($request->password) : $user->password;
        if ($request->has('photo')){
            if (file_exists(public_path().'/'.$user->photo)){
                File::delete(public_path().'/'.$user->photo);
            }
            $data['photo'] = ImageHelper::upload($request->photo, 'users/profile');
        }

        $data['email_receive'] = $request->email_receive ? $request->email_receive : 0;

        $user->update($data);

        return response()->json(['message' => __('members.save_successfully')]);
    }

    public function reservationSearch(Request $request)
    {
        if ($request->ajax()){
            $user = Auth::user();
            $reservations = Reservation::userReservations(null, $request->re_date_from, $request->re_date_to, $request->re_no, $user->id);
            return view($this->view.'reservation_search', compact('reservations', 'user'));
        }
    }

    /**
     * Register form, it's show phone input only
     * @return Factory|View'
     */
    public function register()
    {
        $title = __('main.new_register');
        $countryCods = Country::withDescription()->pluck('code')->toArray();
        return view($this->view.'register', get_defined_vars());
    }

    public function checkRegisterPhone(CheckPhoneRequest $request)
    {
        $user = User::create([
            'phone' => $request->phone,
            'code' => $request->code,
            'role' => 'user',
            'agreement' => 0,
        ]);

        if ($this->checkAttemptCount($request)){
            return $this->checkAttemptCount($request);
        }

        $generateVerifyCode = $this->generateVerifyCode($user);

        if ($generateVerifyCode['errors']){
            return $this->checkAttemptCount($request) ?? response()->json($generateVerifyCode);
        }


        session()->put(['operation_type' => 'register', 'user_phone' => $user->phone]);
        return $this->checkAttemptCount($request) ?? response()->json(['url' => route('user.typeVerifyCode')]);
    }

    public function typeVerifyCode()
    {
        if (!session('user_phone')){
            return redirect()->route('user.register')->with('errors',  'There\'s no code generated');
        }
        $title = __('members.type_verify_code');

        return view($this->view.'type_verify_code', get_defined_vars());
    }

    /**
     * Show register form for user
     * @return Factory|RedirectResponse|View
     */
    public function completeRegister()
    {
        if (session('user_phone') && session('is_verified_code')){
            $title = __('main.complete_register');
            $countryCods = Country::withDescription()->pluck('code')->toArray();
            // Use register_type session
            // to specific which user type is register to show
            // privacy policy and terms page
            session()->put(['register_type' => 'user']);
            return view($this->view.'complete_register', get_defined_vars());
        }

        return redirect()->route('user.register')->with('errors',  'There\'s no code generated');

    }

    /**
     * Register new user
     * @param UserRegisterRequest $request
     * @return JsonResponse
     */
    public function doRegister(UserRegisterRequest $request)
    {
        if (
            $request->ajax() &&
            session('user_phone') &&
            session('is_verified_code') &&
            session('operation_type') == 'register'
        ) {

            $user = User::where('phone', session('user_phone'))->first();
            $data = $request->all();
            $data['password'] = bcrypt($request->password);
            $data['verify_code'] = null;
            $data['code_verify_attempt'] = null;
            $data['role'] = 'user';

            if ($user){
                $user->update($data);
            } else {
                $user = User::create($data);
            }

            Auth::login($user);
            session()->put(['store_device_token' => true]);
            session()->forget(['operation_type', 'user_phone', 'country_code', 'is_verified_code']);
            return response()->json(['url' => route('home')]);
        }
    }

    public function getState(Request $request)
    {
        if ($request->ajax()){
            $states = Region::withDescription($request->country_id);
            if (count($states)){
                return response()->json(['states' => $states]);
            } else {
                return response()->json(['states' => false]);
            }
        }
    }

    /**
     * Check verify code attempt and if it's correct or not
     * @param Request $request
     * @return JsonResponse
     * @throws ValidationException
     */
    public function verifyCode(Request $request)
    {

        $code = implode('',  array_reverse($request->code));

        if (mb_strlen($code) < 4){
            return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.enter_full_code')]);
        }

        $user = User::where('phone', session('user_phone'))->first()
            ->where('verify_code', $code)->first();

        if (!$user){
            return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.invalid_verification_code')]);
        }

        // Check the number of attempts to check the code
        if ($user->code_verify_attempt >= 6 && strtotime($user->last_attempt_date) == strtotime(date('Y-m-d')) ) {
            return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.verification_tries_exceed')]);
        }

        // Check if date less then today
        if ($user->last_attempt_date && strtotime($user->last_attempt_date) < strtotime(date('Y-m-d'))){
            $user->update(['code_verify_attempt' => 0, 'last_attempt_date' => date('Y-m-d')]);
        }

        // The user entered the wrong code and the number of attempts increased
        if (strtotime($user->last_attempt_date) == strtotime(date('Y-m-d')) && $user->verify_code != $code){
            $user->update(['code_verify_attempt' => $user->code_verify_attempt+1]);
            return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.invalid_verification_code')]);
        }

        $user->update(['verify_code' => null]);

        if (session('operation_type') == 'register'){
            $user->update(['email_verified_at' => date('Y-m-d')]);
            session()->put(['country_code' => $user->code, 'is_verified_code' => true]);
            return response()->json([
                'url' => route('user.completeRegister'),
                'message' => __('members.correct_code')
            ]);
        }
        if (session('operation_type') == 'change_phone'){
            $user->update(['phone' => session('new_phone')]);
            session()->forget(['operation_type', 'user_phone', 'new_phone', 'is_verified_code' => true]);
            return response()->json([
                'url' => route('user.profile'),
                'message' => __('members.phone_changed_successfully')
            ]);
        }
    }

    /**
     * check attempt try and generate code
     * @param $user
     * @return array
     */
    private function generateVerifyCode($user)
    {

        /**
         * It's commented for test only and after test
         * delete comment
         */

//        if ($user->attempt_send_code >= 6 && strtotime($user->last_attempt_date) == strtotime(date('Y-m-d')) ) {
//            return ['errors' => __('members.verification_code_exceed')];
//        }

        if ($user->last_attempt_date && strtotime($user->last_attempt_date) < strtotime(date('Y-m-d'))){
            $user->update(['code_verify_attempt' => 0, 'attempt_send_code' => 0,'last_attempt_date' => date('Y-m-d')]);
        }

        $attempt_send_code = $user->attempt_send_code + 1;
        $last_attempt_date = date('Y-m-d');
        $code = rand(1000,9999);

        $phone = $user->phone;
        session()->put(['verify_code' => $code]);

//        try{
//            $message = __('members.sms_message');
//            $message = str_replace('@@', $code, $message);
//            Sms::send('966'.$phone, $message);
//        } catch (Exception $exception){
//            return ['errors' => __('members.sms_error')];
//        }

        /**
         * $data = [
        'user' => $user->name,
        'email' => $user->email,
        'code' => $code,
        ];
        CodeVerificationJob::dispatch($data, 'web.members.emails.verification_email');
         */

        $user->update(
            [
                'verify_code' => $code,
                'attempt_send_code' => $attempt_send_code,
                'last_attempt_date' => $last_attempt_date,
            ]
        );

        return ['errors' => false];
    }


    /**
     * Resend code for register and login
     * @param Request $request
     * @return JsonResponse
     * @throws ValidationException
     */
    public function resendCode(Request $request)
    {
        if ($request->ajax()){
            $user = User::where('phone', session('user_phone'))->first();

            if ($this->checkAttemptCount($request)){
                return $this->checkAttemptCount($request);
            }

            $generateVerifyCode = $this->generateVerifyCode($user);

            if ($generateVerifyCode['errors']){
                return $this->checkAttemptCount($request) ?? response()->json($generateVerifyCode);
            }

            return response()->json(['success' => true]);
        }
    }

    /**
     * Check user attempts count
     * @param $request
     * @return JsonResponse
     * @throws ValidationException
     */
    public function checkAttemptCount($request)
    {
        if (method_exists($this, 'hasTooManyLoginAttempts') &&
            $this->hasTooManyLoginAttempts($request)) {
            $this->fireLockoutEvent($request);
            return $this->sendLockoutResponse($request);
        }
        $this->incrementLoginAttempts($request);

    }

    /**
     * Override parent sendLockoutResponse to handle it
     * to return json response
     * @param Request $request
     * @return JsonResponse
     */
    protected function sendLockoutResponse(Request $request)
    {
        $seconds = $this->limiter()->availableIn(
            $this->throttleKey($request)
        );
        return response()->json(['errors' => Lang::get('auth.throttle', [
            'seconds' => $seconds,
            'minutes' => ceil($seconds / 60),
        ])]);
    }


}
