<?php

namespace App\Http\Controllers\Web;

use App\City;
use App\Hall;
use App\Helpers\QrCode;
use App\Helpers\Sms;
use App\Http\Requests\AddReservationRequest;
use App\Http\Requests\SaveSelectedServicesRequest;
use App\Jobs\InvitationJob;
use App\Jobs\SendAcceptRequestJob;
use App\Mail\InvitationMail;
use App\Occasion;
use App\Region;
use App\Reservation;
use App\Http\Controllers\Controller;
use App\Sector;
use App\Service;
use App\UsersNotification;
use Exception;
use Illuminate\Contracts\View\Factory;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\View\View;


class ReservationController extends Controller
{

    protected $view = 'web.reservations.';

    public function __construct()
    {
        $this->middleware('check_user_operating_system');
    }

    /**
     * Show reservation page to reserve hall
     * @param Hall $hall
     * @return Factory|View
     */
    public function reservationPage(Hall $hall)
    {

        $title = __('reservations.hall_reservation');
        $user = Auth::user();

        if ($user->role != 'user'){
            return  redirect()->route('home');
        }

        if (!$hall->hall_status || !$hall->allow_requests || !$hall->provider){
            return  redirect()->route('hall.index')->with('alert_pop_up', __('halls.reservation_unavailable'));
        }

        $region = Region::withDescription(null, $hall->region_id);
        $city = City::withDescription(null, $hall->city_id);
        $sector = Sector::withDescription(null, $hall->sector_id);

        /**
         * Check if whether region or city or sector not deleted
         */
        if (!$region || !$city || !$sector){
            return redirect()->route('home');
        }

        $uncompletedReservation = Reservation::where('re_no', null)
            ->where('user_id', $user->id)
            ->where('hall_id', $hall->id)->first();

        if ($uncompletedReservation){
            $uncompletedReservation->forceDelete();
        }

        /**
         * used in increment dates in javascript to check date value when date input changed
         */
        $availableDates = [];
        $years = [];
        $months = [];
        $discountDates = [];
        $discountPercentage = [];

        $today = date('Y-m-d');
        $hall->available_dates = $hall->available_dates['dates'];
        if (is_array($hall->available_dates) && count($hall->available_dates)){
            foreach ($hall->available_dates as $key => $date) {
                $date = Carbon::make($date);
                $years[] = $date->format('Y');

                /**
                 * Check old dates and remove it
                 */
                if ($date->format('Y-m-d') < $today){
                    continue;
                }

                /**
                 * TO use it in calender filter
                 */
                $months[(int) $date->format('m').'_'.$date->format('Y')] = $date->format('M');
                $availableDates[$date->format('Y-m-d')] =  0;

                if (count($hall->discount_dates)){
                    foreach ($hall->discount_dates as $discount_date) {
                        if (in_array($date->format('Y-m-d'), $discount_date['dates'])){
                            $availableDates[$date->format('Y-m-d')] =  $discount_date['discount_percentage'];
                        }
                    }
                } else{
                    $availableDates[$date->format('Y-m-d')] =  0;
                }

            }
        }

        if (session('dates_from_search_filter')){
            $selectedDate = session('dates_from_search_filter');
            $years = [];
            $months = [];
            $sessionDiscountPercentage = [];
            foreach ($availableDates as $date => $percentage) {
                if ($date == $selectedDate){
                    $years[] = date('Y', strtotime($date));
//                    $months[] = date('M', strtotime($date));
//                    $months[(int) $date->format('m').'_'.$date->format('Y')] = $date->format('M');
                    $index = date('m', strtotime($date)).'_'.date('Y', strtotime($date));
                    $months[$index] = date('M', strtotime($date));
                    $sessionDiscountPercentage = $availableDates[$date];
                    $availableDates = [$date => $percentage];
                    break;
                }
            }
        }


        $years = array_values(array_unique($years));

        $services = $hall->allow_services ? Service::withDescription(null, $hall->reservation_type) : [];
        $addedServices = is_string($hall->added_services) ? json_decode($hall->added_services, true) : $hall->added_services;
        $occasions = Occasion::withDescription($hall->occasions);

        return view($this->view.'reservation_page', get_defined_vars());
    }

    /**
     * Show reservation details in user profile
     * @param Reservation $reservation
     * @return Factory|RedirectResponse|View
     */
    public function reservationDetails(Reservation $reservation)
    {
        $user = Auth::user();
        $view = null;
        $title = __('reservations.edit');

        if(!$reservation->hall->hall_status){
            return redirect()->route('home');
        }

        // Check for user
        if ($user->role == 'user'){

            $view = 'user_reservation_details';

            if ($reservation->user_id != $user->id){
                return redirect()->route('home');
            }

            if (in_array($reservation->re_status, ['waiting_for_pay', null])){
                return redirect()->route('reservation.payNow', $reservation->id);
            }
        }

        // Check for user
        if ($user->role == 'provider'){

            $view = 'provider_reservation_details';

            if ($reservation->provider_id != $user->id){
                return redirect()->route('home');
            }

            if ($reservation->re_status == null){
                return redirect()->route('home');
            }
        }

        $userReservation = Reservation::userReservations(null, null, null, null, null, $reservation->id);
        $servicesAndAddedServices = $this->getReservationBillDetails($reservation)['details'];
        $hall = $reservation->hall;

        $totalBeforeAddedValue = $reservation->total; // Total after discount and before added value
        $addedValueAmount = $reservation->added_value_amount;
        $totalAfterAddedValue = $totalBeforeAddedValue+$addedValueAmount;
        $providerWebSitePercentage = $reservation->provider->website_percentage;
        $sitePercentageValue = ($totalBeforeAddedValue*$providerWebSitePercentage)/100;

        $reservationDays = $reservation['re_days'];

        $seller = settings('website_name');
        $invoiceDate = $reservation->created_at;
        $invoiceTotalAmount = $reservation->net_total;
        $addedValue = settings('added_value');
        $invoiceTaxAmount = ($invoiceTotalAmount*$addedValue)/100;

        $qrCode = QrCode::generate($seller, $invoiceDate, $invoiceTotalAmount, $invoiceTaxAmount);

        return view($this->view.$view, get_defined_vars());
    }


    /**
     * Create new reservation to hall
     *
     * @param AddReservationRequest $request
     * ## How to calculate reservation number of people
     * (Number of people * Price per person * Number of days) +
     * (Price of the service approved by the platform * service amount) +
     * (Price of the additional service * service amount) = Grand total
     * Net Total = Discounts are applied first and VAT added last
     * ## How to calculate reservation by number of days
     * (Reservation price per day * number of days) +
     * (price of services approved by the platform * service amount) +
     * (Price of additional services * service amount) = Grand total
     * Net Total = Discounts applied first and VAT added last
     * Discount applied foreach (price per day-person, services, added service
     * @param Hall $hall
     * @return JsonResponse
     */
    public function addReservation(AddReservationRequest $request, Hall $hall)
    {

        if ($request->ajax()){
            if (!Auth::check()){
                return response()->json(['url' => route('user.login')]);
            }

            if (!$hall->hall_status || !$hall->allow_requests || !$hall->provider){
                return  response()->json(['errors' => __('halls.reservation_unavailable')]);
            }

//            $lastReservation = Reservation::whereNotNull('re_no')->orderBy('id', 'desc')->first();
//            $re_no = $lastReservation ? ($lastReservation->re_no+1) : '200';
            $user = Auth::user();
            $netTotal = 0;
            $total = 0;
            $minCapacity = $hall->min_capacity;
            $addedValue = settings('added_value') ?? 0;


            $services = request('services') ? json_decode(request('services'), true) : [];
            $addedServices = request('added_services') ? json_decode(request('added_services'), true) : [];

            if (count($services)){
                foreach ($services as $service) {
                    foreach ($service as $oneService) {
                        $price = $oneService['price']*$oneService['quantity'];
                        $netTotal += $price;
                        $discount = ($price*$oneService['discount_percentage'])/100;
                        $price = $price-$discount;
                        $total += $price;
                    }
                }
            }

            if (count($addedServices)){
                foreach ($addedServices as $addedService) {
                    foreach ($addedService as $oneAddedService) {
                        $price = $oneAddedService['price']*$oneAddedService['quantity'];
                        $netTotal += $price;
                        $discount = ($price*$oneAddedService['discount_percentage'])/100;
                        $price = $price-$discount;
                        $total += $price;
                    }
                }
            }


            $userSelectedDays = []; // User-selected days from request
            $isDiscount = []; // Days that have a discount

            $availableDates = is_string($hall->available_dates) ? json_decode($hall->available_dates, true) : $hall->available_dates;
            $allAvailableDates = [];

            if (count($availableDates['dates'])){
                foreach ($availableDates['dates'] as $key => $date) {
                    $date = Carbon::make($date);
                    $allAvailableDates[$date->format('Y-m-d')] =  0;
                    if (count($hall->discount_dates)){
                        foreach ($hall->discount_dates as $discount_date) {
                            if (in_array($date->format('Y-m-d'), $discount_date['dates'])){
                                $allAvailableDates[$date->format('Y-m-d')] =  $discount_date['discount_percentage'];
                            }
                        }
                    } else{
                        $allAvailableDates[$date->format('Y-m-d')] =  0;
                    }

                }
            }

            foreach ($request->re_days as $date) {
                if (array_key_exists($date, $allAvailableDates) || $allAvailableDates[$date]){
                    $isDiscount[] = $allAvailableDates[$date];
                    $userSelectedDays['days'][] = $date;
                    $userSelectedDays['percentage'][] = $allAvailableDates[$date];
                } else {
                    $isDiscount[] = $allAvailableDates[$date];
                    $userSelectedDays['days'][] = $date;
                    $userSelectedDays['percentage'][] = $allAvailableDates[$date];
                }
            }

            /**
             * It's initialize anyway whether there is a discount or not
             * because $isDiscount array contains values
             */
            if (count($isDiscount)){
                // check if person count greater than min capacity
                $personCount = $minCapacity > $request->person_count ? $minCapacity : $request->person_count;
                $personCount = $personCount > $hall->total_capacity ? $hall->total_capacity : $personCount;

                foreach ($isDiscount as $discount) {
                    if ($hall->reservation_type == 'person_count'){
                        $personPrice = ($personCount*$hall->booking_amount_per_person);
                        $netTotal += $personPrice;
                        $personPrice = $personPrice-(($personPrice*$discount)/100);
                        $total += $personPrice;
                    }

                    if ($hall->reservation_type == 'days_count'){
                        $dayPrice = $hall->booking_amount_per_day;
                        $netTotal += $dayPrice;
                        $dayPrice = $dayPrice-(($dayPrice*$discount)/100);
                        $total += $dayPrice;

                    }
                }
            }

            $addedValueAmount = ($total*$addedValue)/100;

            $reservation = Reservation::where('user_id', $user->id)
                ->where('hall_id', $hall->id)->whereNull('re_status')->first();

//            $addedServiceContainer = [];
//            $services = Service::whereIn('services.id', $request->services ?? [])
//                ->join('service_descriptions as sd', 'sd.service_id', 'services.id')
//                ->select([
//                    'services.id',
//                    'services.service_type',
//                    'services.image',
//                    'services.price',
//                    'sd.language_id',
//                    'sd.name',
//                    'sd.description',
//                ])->cursor()->groupBy('language_id')->toArray();
//
//            $addedService = is_string($hall->added_services) ? json_decode($hall->added_services, true) : $hall->added_services;
//
//            $addedServiceContainer = array_filter($addedService, function ($value) use ($request){
//                if (in_array($value['id'], $request->added_services)){
//                    return $value;
//                }
//            });

            $data = [
//                're_no' => $re_no,
                're_way' => $hall->reservation_way,
                'services' => $services, //$request->services ? $request->services : [],
                'added_services' => $addedServices, // $request->added_services ? $request->added_services : [],
                'person_count' => $request->person_count ?? $hall->total_capacity,
                'reservation_type' => $hall->reservation_type,
                'booking_amount_per_person' => $hall->booking_amount_per_person,
                'booking_amount_per_day' => $hall->booking_amount_per_day,
                'net_total' => $netTotal, // Total without any discount or added value
                'total' => $total, // Total + added value and after apply discount
                'other_details' => $request->other_details,
                'added_value_amount' => $addedValueAmount,
                're_days' => $userSelectedDays,
                're_date_from' => date('Y-m-d', strtotime($userSelectedDays['days'][0])),
                're_date_to' => date('Y-m-d', strtotime(array_last($userSelectedDays['days']))),
//                    're_date_from' => date('Y-m-d', strtotime($request->re_date_from)),
//                    're_date_to' => date('Y-m-d', strtotime($request->re_date_to)),
                'user_id' => $user->id,
                'provider_id' => $hall->provider->id,
                'hall_id' => $hall->id,
                'occasion_id' => $request->occasion_id,
            ];


            if ($reservation){
                $reservation->update($data);
            } else {
                $reservation = Reservation::create($data);
                $reNo = (199+$reservation->id); // It's start from 199 because reservation numbers begin from 200
                $reservation->update(['re_no' => $reNo]);
            }

//            if ($isReservedBefore){
//                unset($data['re_no']);
////            date_default_timezone_set('Africa/Cairo');
//                $isReservedBefore->update($data);
//            } else {
//
////            date_default_timezone_set('Africa/Cairo');
//                $reserevation = Reservation::create($data);
//            }

            return response()->json(['url' => route('reservation.payNow', $reservation->id)]);

//            if ($hall->reservation_way == 'direct'){
//                return response()->json(['url' => route('reservation.payNow', $reserevation->id)]);
//            }
//            if ($hall->reservation_way == 'accept_request'){
//                return response()->json(['url' => route('user.profile')]);
//            }
        }

    }

    public function saveSelectedServices(SaveSelectedServicesRequest $request, Hall $hall)
    {
        $user = Auth::user();
        $selectedServices =  [];
        $selectedAddedServices = [];

        $discountPercentage = 0;
        if (count($hall->discount_dates)){
            foreach ($hall->discount_dates as $key => $discount_date) {
                if (in_array($request->selected_day, $discount_date['dates'])){
                    $discountPercentage = $hall->discount_dates[$key]['discount_percentage'];
                }
            }
        }

        if ($request->services && count($request->services)){
            $services = Service::whereIn('services.id', $request->services ?? [])
                ->join('service_descriptions as sd', 'sd.service_id', 'services.id')
                ->where('sd.language_id', currentLanguage()->id)
                ->select([
                    'services.id',
                    'services.service_type',
                    'services.image',
                    'services.price',
                    'sd.language_id',
                    'sd.name',
                    'sd.description',
                ])->cursor()->toArray();

            foreach ($services as $key => $service) {
                $service['quantity'] = $request->services_quantity[$key];
                $service['selected_day'] = $request->selected_day;
                $service['discount_percentage'] = $discountPercentage;
                $selectedServices[$request->selected_day][] = $service;

//                foreach ($service as $key => $item) {
//                    $item['quantity'] = $request->services_quantity[$key];
//                    $item['selected_day'] = $request->selected_day;
//                    $selectedServices[$request->selected_day][] = $item;
//                }
            }

        }

        if ($request->added_services && count($request->added_services)){
            $addedService = is_string($hall->added_services) ? json_decode($hall->added_services, true) : $hall->added_services;
            $addedServices = array_filter($addedService, function ($value) use ($request){
                if (in_array($value['id'], $request->added_services)){
                    return $value;
                }
            });

            $addedServices = array_values($addedServices);

            foreach ($addedServices as $key => $addedSer) {
                $addedSer['quantity'] = $request->added_services_quantity[$key];
                $addedSer['selected_day'] = $request->selected_day;
                $addedSer['discount_percentage'] = $discountPercentage;
                $selectedAddedServices[$request->selected_day][] = $addedSer;
            }

        }

        if (!count($selectedServices) && !count($selectedAddedServices)){
            return response()->json(['errors' => __('reservations.please_select_services')]);
        }

        $reservation = Reservation::where('re_no', null)
            ->where('user_id', $user->id)
            ->where('hall_id', $hall->id)->first();

        if (!$reservation){
            $reservation = Reservation::create([
                'user_id' => $user->id,
                'hall_id' => $hall->id,
                'services' => $selectedServices,
                'added_services' => $selectedAddedServices
            ]);
        } else {
            $reservationServices = array_merge($reservation->services, $selectedServices);
            $reservationAddedServices = array_merge($reservation->added_services, $selectedAddedServices);
            $reservation->update(['services' => $reservationServices, 'added_services' => $reservationAddedServices]);
        }

        return response()->json([
            'message' => __('reservations.data_save_successfully'),
            'services' => $reservation->services,
            'added_services' => $reservation->added_services
        ]);


    }


    /**
     * When pay now button clicked then go to my fatoorah to finish pay
     * @param Reservation $reservation
     * @return Factory|RedirectResponse|View
     */
    public function payNow(Reservation $reservation)
    {
        $title = __('reservations.bill_summary');
        $user = Auth::user();
        if ($user->role != 'user' || $reservation->user_id != $user->id){
            return redirect()->route('home');
        }

        if (in_array($reservation->re_status, ['confirmed', 'reject', 'canceled', 'unconfirmed', 'finished'])){
            return redirect()->route('home');
        }

        $hall = Hall::where('id', $reservation->hall_id)->first();

        $servicesAndAddedServices = $this->getReservationBillDetails($reservation)['details'];

        $totalBeforeAddedValue = $reservation->total; // Total after discount and before added value
        $addedValueAmount = $reservation->added_value_amount;
        $totalAfterAddedValue = $totalBeforeAddedValue+$addedValueAmount;
        $sitePercentageValue = ($totalBeforeAddedValue*settings('website_percentage'))/100;

        $reservationDays = $reservation['re_days'];
        $showApplePay = request('operating_system') == 'APPLE' ? true : false;

        $seller = settings('website_name');
        $invoiceDate = $reservation->created_at;
        $invoiceTotalAmount = $reservation->net_total;
        $addedValue = settings('added_value');
        $invoiceTaxAmount = ($invoiceTotalAmount*$addedValue)/100;

        $qrCode = QrCode::generate($seller, $invoiceDate, $invoiceTotalAmount, $invoiceTaxAmount);

        return view($this->view.'pay_now', get_defined_vars());
    }

    private function getReservationBillDetails($reservation){
        $services = is_string($reservation->services) ? json_decode($reservation->services) : ($reservation->services ?? []);
        $addedServices = is_string($reservation->added_services) ? json_decode($reservation->added_services) : ($reservation->added_services ?? []);
        $reDays = $reservation->re_days;
        $servicesTotal = 0;
        $addedServiceTotal = 0;

//        dd($services, $addedServices, $reDays);

//        $servicesTotal = array_map(function ($value) {
//            return $value['price'];
//        }, $services);
//        $servicesTotal = array_sum($servicesTotal);
//
//        $addedServiceTotal = array_map(function ($value) {
//            return $value['price'];
//        }, $addedServices);
//        $addedServiceTotal = array_sum($addedServiceTotal);

        // ######################################

        $servicesAndAddedServices = [];
        if (count($services)){
            foreach ($services as $day => $service) {
                foreach ($service as $oneService) {
                    $servicesTotal += $oneService['price']*$oneService['quantity'];
                    $servicesAndAddedServices[$day]['services'][] = $oneService;
                }
            }
        }

        if (count($addedServices)){
            foreach ($addedServices as $day => $addedService) {
                foreach ($addedService as $oneAddedService) {
                    $addedServiceTotal += $oneAddedService['price']*$oneAddedService['quantity'];
                    $servicesAndAddedServices[$day]['added_services'][] = $oneAddedService;

                }
            }
        }


        foreach ($reDays['days'] as $key =>  $reDay) {
            if (count($services) && !key_exists($reDay, $services)){
                $servicesAndAddedServices[$reDay]['services'][] = [
                    "id" => null,
                    "service_type" => $reservation->reservation_type,
                    "image" => null,
                    "price" => 0,
                    "language_id" => 1,
                    "name" => null,
                    "description" => null,
                    "quantity" => 0,
                    "selected_day" => $reDay,
                    "discount_percentage" => $reDays['percentage'][$key],
                ];
            }
            if (count($addedServices) && !key_exists($reDay, $addedServices)){
                $servicesAndAddedServices[$reDay]['added_services'][] = [
                    "id" => null,
                    "service_type" => $reservation->reservation_type,
                    "image" => null,
                    "price" => 0,
                    "language_id" => 1,
                    "name" => null,
                    "description" => null,
                    "quantity" => 0,
                    "selected_day" => $reDay,
                    "discount_percentage" => $reDays['percentage'][$key],
                ];
            }
        }


        ksort($servicesAndAddedServices);

        return ['details' => $servicesAndAddedServices];

    }

    /**
     * Send reservation request when hall reservation way = accept_request
     * @param Reservation $reservation
     * @return JsonResponse
     */
    public function sendReservationRequest(Reservation $reservation)
    {
        $route = route('user.profile');
        $reservation->update(['re_status' => 'unconfirmed']);
        $provider = $reservation->provider;

        /** Start notification block */
        $userReservation = Reservation::userReservations(null, null, null, null, null, $reservation->id);
        $title = __("reservations.reservation_unconfirmed_title") . "($userReservation->hall_name)";
        $message = __("reservations.reservation_unconfirmed_message");
        $messageForProvider = str_replace_array('@@', [$userReservation->provider_name, $userReservation->hall_name, $userReservation->client_name], $message);
        $emailData = ['title' => $title, 'user' => $provider->name, 'message' => $messageForProvider, 'email' => $provider->email];

        try{
            SendAcceptRequestJob::dispatch($emailData, "web.members.emails.send_accept_request_email");
        } catch (\Exception $e){}


        $class = 'alertItem';
        $icon = 'noti-alert.svg';
        $containerId = '#pills-uncertain';
        $badge = 'badge_uncertain';
        $badgeTitle = __('reservations.unconfirmed');
        $notification_type = 'alert';



//            date_default_timezone_set('Africa/Cairo');

        $providerNotification = UsersNotification::create([
            'title' => $title,
            'message' => $messageForProvider,
            'user_id' => $userReservation->provider_id,
            'notification_type' => $notification_type,
            'page_link' => route('reservation.reservationDetails',$reservation->id),
        ]);

        $icon = asset("assets/web/images/icons/profile/$icon");

        $now = \Carbon\Carbon::now();
        $notificationTime = Carbon::make($reservation->created_at);
        $notificationTime = $now->diffInMinutes($notificationTime);
        $data = [
            'reservation_done' => 'done',
            'reservation_route' => route('reservation.reservationDetails',$reservation->id),
            'reservation_image' => asset(json_decode($userReservation->images, true)[0]),
            'hall_name' => $userReservation->hall_name,
            're_no' => "$userReservation->re_no",
            're_date_from' => date('d/m/Y', strtotime($userReservation->re_date_from)),
            'person_count' => "$userReservation->person_count",
            'total' => "$userReservation->total",
            'user_photo' => asset($user->photo ?? 'assets/web/images/avatar.png'),
            'notification_time' => $notificationTime . __('halls.minutes') . __('halls.ago'),
            'notification_date' => date('M Y d', strtotime($reservation->created_at)),
            'notification_icon' => $icon,
            'notification_class' => $class,
            'reservation_container_id' => $containerId,
            'badge' => $badge,
            'badge_title' => $badgeTitle,
            'change_reservation_status' => 'yes',
        ];

        $providerData = array_merge([
            'name' => $userReservation->provider_name,
            'notification_title' => $title,
            'notification_message' => $messageForProvider,
            'notification_id' => "$providerNotification->id"
        ], $data);


        if (is_array($provider->device_token) && isset($provider->device_token['token'])){
            $notificationForProvider = new SendNotification($title, $messageForProvider, $icon, $providerData, route('provider.profile'));
            $provider->notify($notificationForProvider);
        }


        return response()->json(['message' => __('reservations.request_send_successfully'), 'route' => $route, 'send_request' => true]);
    }

    /**
     * Change reservation status foreach user and provider
     * @param Request $request
     * @param $reservation
     * @return JsonResponse
     */
    public function reservationStatusOperation(Request $request, $reservation)
    {
        if ($request->ajax()){
            $reservation = Reservation::find($reservation);
            $authUser = Auth::user();
            $user = $reservation->user;
            $provider = $reservation->provider;
            $errors = [];
            $cancelledFrom = null;
            $reStatus = null;

            if ($authUser->role == 'user'){
                $cancelledFrom = 'user';
                $reStatus = 'canceled';
                if (!$reservation || $reservation->user_id != $authUser->id){
                    $errors[] = __('reservations.reservation_not_found');
                }
            }
            if ($authUser->role == 'provider'){
                if (!$reservation || $reservation->provider_id != $authUser->id){
                    $errors[] = __('reservations.reservation_not_found');
                }
                if (!in_array( $request->re_status, ['canceled', 'reject', 'waiting_for_pay'])){
                    $errors[] = __('reservations.unknown_reservation_status');
                }
                $reStatus = $request->re_status;
                $cancelledFrom = in_array($request->re_status, ['canceled', 'reject']) ? 'provider' : null;
            }

            if (time() >= strtotime($reservation->re_date_from)){
                $errors[] =  __('reservations.cannot_do_reservation_action');
            }

            if (count($errors)){
                return response()->json(['errors' => $errors[0]]);
            }

            $reservation->update(['re_status' => $reStatus, 'canceled_from' => $cancelledFrom]);

            /** Start notification block */
            $userReservation = Reservation::userReservations(null, null, null, null, null, $reservation->id);
            $title = __("reservations.reservation_{$reStatus}_title") . "($userReservation->hall_name)";
            $message = __("reservations.reservation_{$reStatus}_message");
            $messageForUser = str_replace_array('@@', [$userReservation->client_name, $userReservation->hall_name, $userReservation->provider_name], $message);
            $messageForProvider = str_replace_array('@@', [$userReservation->provider_name, $userReservation->hall_name, $userReservation->provider_name], $message);

            $class = '';
            $icon = '';
            $containerId = '';
            $badge = '';
            $badgeTitle = '';
            $notification_type = null;
            if ($reStatus == 'waiting_for_pay'){
                $notification_type = 'alert';
                $class = 'alertItem';
                $icon = 'noti-alert.svg';
                $containerId = '#pills-pinding';
                $badge = 'badge_pinding';
                $badgeTitle = __('reservations.waiting_for_pay');
            }
            if (in_array($reStatus, ['canceled', 'reject'])){
                $notification_type = 'cancel';
                $class = 'canceledItem';
                $icon = 'noti-canceled.svg';
                $containerId = '#pills-canceled';
                $badge = 'badge_canceled';
                $badgeTitle = __('reservations.canceled');
            }


            $userNotification= UsersNotification::create([
                'title' => $title,
                'message' => $messageForUser,
                'user_id' => $userReservation->client_id,
                'notification_type' => $notification_type,
                'page_link' => route('reservation.reservationDetails',$reservation->id),
            ]);
            $providerNotification = UsersNotification::create([
                'title' => $title,
                'message' => $messageForProvider,
                'user_id' => $userReservation->provider_id,
                'notification_type' => $notification_type,
                'page_link' => route('reservation.reservationDetails',$reservation->id),
            ]);

            $icon = asset("assets/web/images/icons/profile/$icon");
            $now = \Carbon\Carbon::now()->setTimezone('Asia/riyadh');
            $notificationTime = Carbon::make($userNotification->created_at)->setTimezone('Asia/riyadh');
            $notificationTime = $now->diffInMinutes($notificationTime);
            $data = [
                'reservation_done' => 'done',
                'reservation_route' => route('reservation.reservationDetails',$reservation->id),
                'reservation_image' => asset(json_decode($userReservation->images, true)[0]),
                'hall_name' => $userReservation->hall_name,
                're_no' => "$userReservation->re_no",
                're_date_from' => date('d/m/Y', strtotime($userReservation->re_date_from)),
                'person_count' => "$userReservation->person_count",
                'total' => "$userReservation->total",
                'user_photo' => asset($user->photo ?? 'assets/web/images/avatar.png'),
                'notification_time' => $notificationTime . __('halls.minutes') . __('halls.ago'),
                'notification_date' => date('M Y d', strtotime($reservation->created_at)),
                'notification_icon' => $icon,
                'notification_class' => $class,
                'reservation_container_id' => $containerId,
                'badge' => $badge,
                'badge_title' => $badgeTitle,
                'change_reservation_status' => 'yes',
            ];
            $userData = array_merge([
                'name' => $user->name,
                'notification_title' => $title,
                'notification_message' => $messageForUser,
                'notification_id' => "$userNotification->id"
            ], $data);
            $providerData = array_merge([
                'name' => $userReservation->provider_name,
                'notification_title' => $title,
                'notification_message' => $messageForProvider,
                'notification_id' => "$providerNotification->id"
            ], $data);

            if (is_array($user->device_token) && isset($user->device_token['token'])){
                $notificationForUser = new SendNotification($title, $messageForUser, $icon, $userData, route('user.profile'));
                $user->notify($notificationForUser);
            }

            if (is_array($user->device_token) && isset($user->device_token['token'])){
                $notificationForProvider = new SendNotification($title, $messageForProvider, $icon, $providerData, route('provider.profile'));
                $provider->notify($notificationForProvider);
            }

            // waiting_for_pay => accept, [reject, canceled]
//            NotificationHelper::sendNotification('', '');

            return response()->json(['message' => __('reservations.reservation_action_done')]);

        }
    }

    /**
     * Send invitation by email or phone
     * @param Request $request
     * @param $reservationId
     * @return JsonResponse
     */
    public function invitation(Request $request, $reservationId)
    {
        if ($request->ajax()){
            $data = [];
            $user =  Auth::user();
            $languageId = currentLanguage()->id;
            $reservation = Reservation::where('reservations.id', $reservationId)
                ->where('reservations.user_id', $user->id)->where('reservations.re_status', 'confirmed')
                ->join('halls', 'halls.id', 'reservations.hall_id')
                ->join('hall_descriptions as hall_desc', 'halls.id', 'hall_desc.hall_id')
                ->where('hall_desc.language_id', $languageId)
                ->join('occasions', 'occasions.id', 'reservations.occasion_id')
                ->join('occasion_descriptions as occ_desc', 'occasions.id', 'occ_desc.occasion_id')
                ->where('occ_desc.language_id', $languageId)
                ->join('users', 'users.id', 'reservations.user_id')
                ->select([
                    'reservations.*',
                    'halls.street',
                    'halls.lat',
                    'halls.lng',
                    'halls.total_capacity',
                    'hall_desc.name as hall_name',
                    'occ_desc.name as occasion_name',
                    'users.name as user_name',
                ])
                ->first();


            if ($reservation->person_count == $reservation->invitation_count){
                return response()->json(['errors' => __('reservations.invitation_limit')]);
            }

            $emailRule = $request->email ? 'required|email' : 'nullable';
            $phoneRule = $request->phone ? 'required|regex:/^(5){1}([0-9]{8})$/' : 'nullable';

            $validation = Validator::make($request->all(),
                [
                    'email' => $emailRule,
                    'invitation_template' => 'nullable|in:template_1,template_2,template_3',
                    'phone' => $phoneRule,
                ]
                , [], [
                    'email' => __('reservations.email'),
                    'phone' => __('reservations.phone'),
                    'invitation_template' => __('reservations.invitation_template'),
                ]);

            if ($validation->fails()){
                return response()->json(['errors' => $validation->errors()->first()]);
            }

            if ($request->email){

                if (!$request->invitation_template){
                    return response()->json(['errors' => __('reservations.invitation_template_required')]);
                }

                $data['occasion_name'] = $reservation->occasion_name;
                $data['user_name'] = $reservation->user_name;
                $data['email'] = $request->email;
                $data['hall_name'] = $reservation->hall_name;
                $data['date'] = date('l, M d', strtotime($reservation->re_date_from));
                $data['day'] = dateInfo($reservation->re_date_from, ['day_name' => true]);
                $data['city'] = City::withDescription(null, $reservation->hall->city_id)->name ?? '';
                $data['hall_location'] = $location = "https://www.google.com/maps/search/?api=1&query=$reservation->lat,$reservation->lng";
                $data['direction'] = getCurrentLocale() == 'ar' ? 'rtl' : 'ltr';
                $data['address'] = $reservation->street;
                $data['invitation_template'] = $request->invitation_template;
                try{
                    InvitationJob::dispatch($data, "web.members.emails.invitations.{$request->invitation_template}.index");
                } catch (\Exception $e){
                    return  response()->json(['errors' => $e->getMessage()]);
                }

//                // Check if templates invitation directory is exist
//                if(! File::isDirectory(public_path("assets/web/images/invitations/template/template"))){
//                    File::makeDirectory(public_path("assets/web/images/invitations/template/template"), '0755', true, true);
//                }
//
//                // Check if generated invitation directory is exist
//                if(! File::isDirectory(public_path("uploads/invitations/reservation_{$reservationId}"))){
//                    File::makeDirectory(public_path("uploads/invitations/reservation_{$reservationId}"), '0755', true, true);
//                }
//
//                // Check if generated invitation file is exist
//                $invitationTemplateFile = "uploads/invitations/reservation_{$reservationId}/{$request->invitation_template}.png";
//                if(! File::exists(public_path($invitationTemplateFile))){
//                    $userName = $course->user_name;
//                    $userNameImage = generateUserNameAndCourseNameImage($userName, 'user_name', '#ff4f0e');
//                    $userNameTextWidth = $userNameImage->width();
//
//                    $courseName = $course->course_name;
//                    $courseImage = generateUserNameAndCourseNameImage($courseName, 'course_name', '#13608b');
//                    $courseNameTextWidth = $courseImage->width();
//
//
//                    $img = Image::make(public_path('assets/web/images/certification/certification.png'));
//                    $userNameTextWidth = $userNameTextWidth > $img->width() ? $img->width() : $userNameTextWidth;
//                    $courseNameTextWidth = $courseNameTextWidth > $img->width() ? $img->width() : $courseNameTextWidth;
//                    $userNameTextWidth = (int) round(($img->width()-$userNameTextWidth)/2);
//                    $courseNameTextWidth = (int) round(($img->width()-$courseNameTextWidth)/2);
//
//                    $img->insert(public_path('/assets/web/images/certification/template/user_name.png'), 'top-center', $userNameTextWidth, 480)
//                        ->insert(public_path('/assets/web/images/certification/template/course_name.png'), 'top-center', $courseNameTextWidth, 630);
//
//
//                    $cerName = time().'_'.\Illuminate\Support\Str::random(10).'_'.$course->user_name.'.png';
//                    certification($img, 500, 540, 22, '#003322', __('courses.cert_title'));
//                    certification($img,500, 762, 20, '#003322', __('courses.by') . $course->hours.__('courses.training_hours'));
//                    certification($img, 860, 950, 25, '#ff4f0e', $course->instructor_name);
//                    certification($img, 295, 950, 25, '#ff4f0e', $programManager);
//
//                    $path = 'uploads/certifications/'.$cerName;
//                    $img->save(public_path($path));
//                }
//
//                dd('ok');


            }

            if ($request->phone){
                $location = "https://maps.google.com/?q=$reservation->lat,$reservation->lng";
    //                $location = "https://www.google.com/maps/@$reservation->lat,$reservation->lng,20z";
//                $location = "https://www.google.com/maps/search/?api=1&query=$reservation->lat,$reservation->lng";
                $message = __('reservations.invitation_title') .
                    $reservation->hall_name .
                    __('reservations.invite_by') .
                    $user->name . __('reservations.hall_location') . ':' . $location;
                try{ Sms::send('966'.$request->phone, $message); } catch (Exception $e){}
            }



            $invitations = $reservation->invitations ? $reservation->invitations : [];
            $invitations[] = $request->email ?? $request->phone;
            $reservation->update([
                'invitation_count' => ($reservation->invitation_count+1),
                'invitations' => $invitations,
            ]);

            return response()->json(['message' => __('reservations.invitation_successfully'), 'email_phone' => $request->email ?? $request->phone]);
        }
    }
}
