<?php

namespace App\Http\Controllers\Web;

use App\Category;
use App\City;
use App\Contract;
use App\Country;
use App\Hall;
use App\HallDescription;
use App\Helpers\ImageHelper;
use App\Http\Requests\CheckPhoneRequest;
use App\Http\Requests\ProviderHallRequest;
use App\Http\Requests\ProviderRegisterRequest;
use App\Http\Requests\ProviderUpdateProfileRequest;
use App\Jobs\SendAcceptProviderJob;
use App\Occasion;
use App\Region;
use App\Reservation;
use App\Sector;
use App\Table;
use App\User;
use App\UsersNotification;
use Carbon\Carbon;
use Illuminate\Contracts\View\Factory;
use Illuminate\Foundation\Auth\ThrottlesLogins;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\View\View;

class ProviderController extends Controller
{

    use ThrottlesLogins;

    /**
     * The maximum number of attempts to allow.
     *
     * @return int
     */
    protected $maxAttempts = 3;


    /**
     * The number of minutes to throttle for.
     *
     * @return int
     */
    protected $decayMinutes = 1;


    /**
     * The field name throttle trait will use for.
     *
     * @return string
     */
    public function username() : string
    {
        return 'phone';
    }


    protected $view = 'web.members.providers.';

    /**
     * User profile
     * @param Request $request
     * @return Factory|View
     */
    public function index(Request $request)
    {
        $user = Auth::user();

        if (!in_array($user->role, ['provider'])){
            return  abort(404);
        }

        $title = __('members.my_profile');
        $categories = Category::withDescription();
        $regions = Region::withDescription();
        $cities = City::withDescription($user->region_id);
        $sectors = Sector::withDescription($user->city_id);
        $occasions = Occasion::withDescription();
        $tables = Table::withDescription();

        $countryCods = Country::withDescription()->pluck('code')->toArray();
//        $sitePercentage = array_key_exists('website_percentage', settings()) ? settings('website_percentage') : 15;
        $sitePercentage = $user->website_percentage ? $user->website_percentage : 15;
        $languageId = currentLanguage()->id;
        $local = getCurrentLocale();

        $wallets = DB::table('reservations')
            ->where('reservations.re_status', '=', 'confirmed')
            ->whereNull('reservations.deleted_at')
            ->where('provider_id', '=', $user->id)
            ->where('is_paid', '=', 0)
            ->join('halls', 'reservations.hall_id', 'halls.id')
            ->join('hall_descriptions as hall_desc', 'hall_desc.hall_id', 'halls.id')
            ->where('hall_desc.language_id', $languageId)
            ->join('users', 'users.id', 'reservations.user_id')
            ->select(
                [
                    'reservations.*',
                    'halls.images',
                    'hall_desc.name as hall_name',
                    'users.name as client_name',
                    'users.phone as client_phone',
                ]
            )
            ->selectSub("SELECT SUM(total) FROM reservations WHERE is_paid = 0 AND provider_id = {$user->id}", 'wallet_total')
            ->selectSub("SELECT SUM(added_value_amount) FROM reservations WHERE is_paid = 0 AND provider_id = {$user->id}", 'wallet_added_value_amount')->paginate(5);

        $walletTotal = count($wallets) ? ($wallets->first()->wallet_total) : 0;
        $walletTotal = $walletTotal-(($walletTotal*$sitePercentage)/100);

        $halls = Hall::withDescription($user->id, null, null, 1)->groupBy('reservation_type');
        $reservations = Reservation::userReservations(null, null, null, null, null, null, null, $user->id)->groupBy('re_status');
        $dailyReservations = Reservation::userReservations('daily', null, null, null, null, null, null, $user->id);


        $today = date('Y-m-d');

        $arrivals = DB::table('reservations')
            ->whereNull('reservations.deleted_at')
            ->where('reservations.provider_id', '=', $user->id)
            ->join('users as providers', 'providers.id', '=', 'reservations.provider_id')
            ->join('users', 'users.id', '=', 'reservations.user_id')
            ->where('reservations.re_status', '=', 'confirmed')
            ->where('reservations.re_date_from', '=',  $today)
            ->select(['reservations.*', 'providers.name as provider_name', 'users.phone as client_phone'])
            ->cursor();

        $departures = DB::table('reservations')
            ->whereNull('reservations.deleted_at')
            ->where('reservations.provider_id', '=', $user->id)
            ->join('users as providers', 'providers.id', '=', 'reservations.provider_id')
            ->join('users', 'users.id', '=', 'reservations.user_id')
            ->where('reservations.re_status', '=', 'confirmed')
            ->where('reservations.re_date_to', '=',  $today)
            ->select(['reservations.*', 'providers.name as provider_name', 'users.phone as client_phone'])
            ->cursor();

        $notifications = UsersNotification::orderBy('id', 'DESC')
            ->where('user_id', $user->id)
            ->orWhere('user_id', null)
            ->cursor();

        $notificationsCount = [];

        $allNotifications = $notifications->pluck('is_read')->toArray();

        foreach ($allNotifications as $allNotification) {
            if (is_array($allNotification) && !in_array($user->id, $allNotification)){
                $notificationsCount[] = true;
            }
            if (!is_array($allNotification)){
                $notificationsCount[] = true;
            }

        }

        $notificationsCount = count($notificationsCount);

        $contractText = null;
        if ($user->status){
            $contractText = Contract::withDescription()->text;
        }


        return view($this->view.'profile', get_defined_vars());
    }

    public function getWalletByPaginator(Request $request)
    {
        $user = Auth::user();
        $languageId = currentLanguage()->id;
        $wallets = DB::table('reservations')
            ->where('reservations.re_status', '=', 'confirmed')
            ->whereNull('reservations.deleted_at')
            ->where('provider_id', '=', $user->id)
            ->where('is_paid', '=', 0)
            ->join('halls', 'reservations.hall_id', 'halls.id')
            ->join('hall_descriptions as hall_desc', 'hall_desc.hall_id', 'halls.id')
            ->where('hall_desc.language_id', $languageId)
            ->join('users', 'users.id', 'reservations.user_id')
            ->select(
                [
                    'reservations.*',
                    'halls.images',
                    'hall_desc.name as hall_name',
                    'users.name as client_name',
                    'users.phone as client_phone',
                ]
            )
            ->selectSub("SELECT SUM(total) FROM reservations WHERE is_paid = 0 AND provider_id = {$user->id}", 'wallet_total')
            ->selectSub("SELECT SUM(added_value_amount) FROM reservations WHERE is_paid = 0 AND provider_id = {$user->id}", 'wallet_added_value_amount')->paginate(5);

        $sitePercentage = $user->website_percentage ? $user->website_percentage : 15;



        return view($this->view.'get_wallet_paginator', compact('wallets', 'sitePercentage'))->render();
    }

    public function updateProfile(ProviderUpdateProfileRequest $request, User $user)
    {
        $data = $request->all();

        $data['password'] = $request->password ? bcrypt($request->password) : $user->password;
        if ($request->has('photo')){
            $data['photo'] = ImageHelper::upload($request->photo, 'users/profile', $user->photo);
        }
        if ($request->has('logo')){
            $data['logo'] = ImageHelper::upload($request->logo, 'users/logo', $user->logo);
        }

        $data['email_receive'] = $request->email_receive ? $request->email_receive : 0;
        $data['bank_acc_info'] = [
            'bank_name' => $request->bank_name,
            'iban' => $request->iban,
            'acc_no' => $request->acc_no,
            'client_name' => $request->client_name,
        ];

        $user->update($data);

        return response()->json(['message' => __('members.save_successfully')]);
    }

    public function deleteHall(Request $request, $hall_id)
    {
        if ($request->ajax()){
            $hall = Hall::where('id', $hall_id)->where('hall_status', 1)->first();
            $user = Auth::user();
            if (!$hall || $hall->user_id != $user->id){
                return ['errors' => __('halls.hall_not_found')];
            }
            $hall->delete();
            return ['message' => __('halls.hall_delete_successfully')];
        }
    }

    public function editHall(Hall $hall)
    {
        $user = Auth::user();
        $title = __('halls.edit');
        if ($user->role != 'provider' || $hall->user_id != $user->id || !$hall->hall_status){
            return redirect()->route('home');
        }

        $myHall = Hall::withDescription($user->id, $hall->id);

        $regions = Region::withDescription();
        $cities = City::withDescription($myHall->region_id);
        $sectors = Sector::withDescription($myHall->city_id);
        $categories = Category::withDescription();
        $occasions = Occasion::withDescription();
        $tables = Table::withDescription();
        $local = getCurrentLocale();
        $addedServices = $myHall->added_services;
        $addedServices = json_decode($addedServices, true);
        return view($this->view.'edit_hall', get_defined_vars());
    }

    public function AddEditHall(ProviderHallRequest $request, $hall = null)
    {
        if ($request->ajax()){
            $data = $request->all();
            $added_services = [];
            $available_dates = [];
            $data['user_id'] = Auth::user()->id;
            $newImages = [];

            $datesRang = []; // Equal available dates after filter it
            $allUnavailableDates = [];
            $unavailableDates = [];
            $discountDates = [];

            if ($request->added_services_name && count($request->added_services_name)){
                foreach ($request->added_services_name as $key => $addedServiceName) {
                    if ($request->added_services_image[$key] ?? null){
                        $file = ImageHelper::upload($data['added_services_image'][$key]);
                        if ($file['errors'] ?? null){
                            return response()->json(['errors' => $file['errors']]);
                        }
                        $newImage = $file;
                    } else {
                        $newImage = $request->old_added_services_image[$key] ?? null;
                    }

                    array_push($added_services, [
                        'id' => ($key+1),
                        'name' => $addedServiceName,
                        'price' => $request->added_services_price[$key],
                        'description' => $request->added_services_description[$key],
                        'image' => $newImage,
                    ]);
                }
            }


//            if ($request->old_added_services_image){
//                foreach ($data['old_added_services_image'] as $key => $image) {
//                    if ($image == 'new_image_added'){
//                        $file = ImageHelper::upload($data['added_services_image'][$key]);
//                        if ($file['errors'] ?? null){
//                            return response()->json(['errors' => $file['errors']]);
//                        }
//                        $newImage = $file;
//                    } else {
//                        $newImage = $image;
//                    }
//
//                    array_push($added_services, [
//                        'id' => ($key+1),
//                        'name' => $request->added_services_name[$key],
//                        'price' => $request->added_services_price[$key],
//                        'description' => $request->added_services_description[$key],
//                        'image' => $newImage,
//                    ]);
//                }
//            }


            if ($request->images) {
                foreach ($request->images as $image) {
                    $file = ImageHelper::upload($image);
                    if ($file['errors'] ?? null){
                        return response()->json(['errors' => $file['errors']]);
                    }
                    $newImages[] = $file;

                }
            }

            if ($request->old_images){
                $newImages = array_merge($request->old_images, $newImages);
            }

            $data['images'] = $newImages;

            // Generate days between from => to
            $start = Carbon::make($request->dates_rang_from)->addDays(-1);
            $end = Carbon::make($request->dates_rang_to);
            $diff = $end->diff($start)->days;
            if ($diff){
                while($end->diff($start)->days){
                    $start = $start->addDay();
                    $datesRang[] = $start->format('Y-m-d');
                }
            }

            if ($request->unavailable_dates_from){

                foreach ($request->unavailable_dates_from as $fromDateKey => $datesFrom) {
                    $from = Carbon::make($datesFrom)->addDays(-1);
                    $to = Carbon::make($request->unavailable_dates_to[$fromDateKey]);
                    $diff = $to->diff($from)->days;
                    if ($diff){
                        while($to->diff($from)->days){
                            // To filter days and get available dates
//                            if (in_array($from->format('Y-m-d'), $datesRang)){
//                                $key = array_search($from->format('Y-m-d'), $datesRang);
//                                unset($datesRang[$key]);
//                            }
                            $from = $from->addDay();
                            $unavailableDates[$fromDateKey][] = $from->format('Y-m-d');
                            // $allUnavailableDates used to check if it contains in discount dates
                            $allUnavailableDates[] = $from->format('Y-m-d');
                        }
                    }

                }

            }

            if ($request->discount_dates_from){
                foreach ($request->discount_dates_from as $discFromDateKey => $discDatesFrom) {
                    $from = Carbon::make($discDatesFrom)->addDays(-1);
                    $to = Carbon::make($request->discount_dates_to[$discFromDateKey]);
                    $diff = $to->diff($from)->days;
                    if ($diff){
                        while($to->diff($from)->days){
                            $from = $from->addDay();
                            if (in_array($from->format('Y-m-d'), $allUnavailableDates)){
                                return response()->json(['errors' => __('halls.unavailable_dates_cant_be_discount_dates')]);
                            }
                            $discountDates[$discFromDateKey]['dates'][] = $from->format('Y-m-d');
                        }
//                        $discountDates[$discFromDateKey]['discount'] = $request->discount_dates[$discFromDateKey];
                        $discountDates[$discFromDateKey]['discount_percentage'] = $request->discount_percentage[$discFromDateKey] == 'empty' ? 0 : $request->discount_percentage[$discFromDateKey];
                    }

                }
            }

            $data['unavailable_dates'] = $unavailableDates;
            $data['available_dates'] = [
                'dates_rang_from' => $request->dates_rang_from,
                'dates_rang_to' => $request->dates_rang_to,
                'dates' => array_values(array_diff($datesRang, $allUnavailableDates))
            ];
            $data['discount_dates'] = $discountDates;
            $data['added_services'] = json_encode($added_services);

            $languageId = currentLanguage()->id;
            $local = getCurrentLocale();

            if ($request->update_hall_flag){
                $hall = Hall::find($hall);
                $hallDescription = HallDescription::where('hall_id', $hall->id)->where('language_id', $languageId)->first();
                $hall->update($data);
                $hallDescription->update(
                    [
                        'name' => $data['name_'.$local],
                        'description' => $data['description_'.$local],
                        'reservation_condition' => $data['reservation_condition_'.$local],
                        'features' => $data['features_'.$local],
                    ]
                );

            } else {
                $createdHall = Hall::create($data);
                foreach (languages() as $language) {
                    HallDescription::create(
                        [
                            'language_id' => $language->id,
                            'hall_id' => $createdHall->id,
                            'name' => $data['name_'.$local],
                            'description' => $data['description_'.$local],
                            'reservation_condition' => $data['reservation_condition_'.$local],
                            'features' => $data['features_'.$local],
                        ]
                    );
                }

            }
        }
        return response()->json(['url' => route('provider.profile')]);


    }

    public function walletSearch(Request $request)
    {
        if ($request->ajax()){
            $user = Auth::user();
            $reservations = Reservation::userReservations(
                'confirmed',
                $request->re_date_from,
                $request->re_date_to,
                null,
                null,
                null,
                $request->is_paid,
                $user->id
            );

            $sitePercentage = array_key_exists('website_percentage', settings()) ? settings('website_percentage') : 15;
            return view($this->view.'wallet_search', compact('reservations', 'user', 'sitePercentage'));
        }
    }

    public function reservationSearch(Request $request)
    {
        if ($request->ajax()){
            $user = Auth::user();
            $reservations = Reservation::userReservations(
                null,
                $request->re_date,
                null,
                $request->re_no,
                null,
                null,
                null,
                $user->id
            );


            $sitePercentage = array_key_exists('website_percentage', settings()) ? settings('website_percentage') : 15;
            return view($this->view.'reservation_search', compact('reservations', 'user', 'sitePercentage'));
        }
    }


    /**
     * Register form, it's show phone input only
     * @return Factory|View'
     */
    public function register()
    {
        $title = __('main.new_register');
        $countryCods = Country::withDescription()->pluck('code')->toArray();
        return view($this->view.'register', get_defined_vars());
    }

    public function checkRegisterPhone(CheckPhoneRequest $request)
    {
        $user = User::create([
            'phone' => $request->phone,
            'code' => $request->code,
            'role' => 'provider',
            'agreement' => 0,
        ]);

        if ($this->checkAttemptCount($request)){
            return $this->checkAttemptCount($request);
        }

        $generateVerifyCode = $this->generateVerifyCode($user);

        if ($generateVerifyCode['errors']){
            return $this->checkAttemptCount($request) ?? response()->json($generateVerifyCode);
        }


        session()->put(['operation_type' => 'register', 'provider_phone' => $user->phone]);
        return $this->checkAttemptCount($request) ?? response()->json(['url' => route('provider.typeVerifyCode')]);
    }

    public function typeVerifyCode()
    {
        if (!session('provider_phone')){
            return redirect()->route('provider.register')->with('errors',  'There\'s no code generated');
        }
        $title = __('members.type_verify_code');

        return view($this->view.'type_verify_code', get_defined_vars());
    }

    /**
     * Show register form for user
     * @return Factory|RedirectResponse|View
     */
    public function completeRegister()
    {
        if (session('provider_phone') && session('is_verified_code')){
            $title = __('main.new_register');
            $countryCods = Country::withDescription()->pluck('code')->toArray();
            $regions = Region::withDescription();
            // Use register_type session
            // to specific which user type is register to show
            // privacy policy and terms page
            session()->put(['register_type' => 'provider']);
            return view($this->view.'complete_register', get_defined_vars());
        }

        return redirect()->route('provider.register')->with('errors',  'There\'s no code generated');

    }


    /**
     * Register new user
     * @param ProviderRegisterRequest $registerRequest
     * @return JsonResponse
     */
    public function doRegister(ProviderRegisterRequest $registerRequest)
    {
        if (
            $registerRequest->ajax() &&
            session('provider_phone') &&
            session('is_verified_code') &&
            session('operation_type') == 'register'
        ) {
            $user = User::where('phone', session('provider_phone'))->first();
            $data = $registerRequest->all();
            $data['password'] = bcrypt($registerRequest->password);
            $data['role'] = 'provider';
            $data['status'] = 0;
            $data['commercial_file'] = ImageHelper::upload($registerRequest->commercial_file, 'provider/commercial_file');
            $data['logo'] = ImageHelper::upload($registerRequest->logo, 'provider/logo');

            if ($user){
                $user->update($data);
            } else {
                $user = User::create($data);
            }

            try{
                $data = [
                    'email' => $user->email,
                    'user' => $user->name,
                    'title' => __('members.accept_provider_title'),
                    'message' => __('members.register_data_review_message'),
                ];
                SendAcceptProviderJob::dispatch($data, 'web.members.providers.emails.accept_provider'); // himabehiss@gmail.com
            } catch(\Exception $exception){}
            Auth::login($user);
            session()->put(['store_device_token' => true]);
            session()->forget(['operation_type', 'provider_phone', 'country_code', 'is_verified_code']);
            return response()->json(['url' =>route('provider.profile')]);
        }
    }

    public function getCityAndSector(Request $request)
    {
        if ($request->ajax()){
            $cities = [];
            $sectors = [];
            if ($request->region_id){
                $cities = City::withDescription($request->region_id);
            }

            if ($request->city_id){
                $sectors = Sector::withDescription($request->city_id);
            }
            return ['get_type' => $request->get_type, 'cities' => $cities, 'sectors' => $sectors];
        }
    }


    /**
     * Update contract agreement
     * @param Request $request
     * @param User $user
     * @return array
     */
    public function contractAgreement(Request $request, User $user)
    {
        $validation = Validator::make($request->all(), [
            'contract_agreement' => 'required|in:1'
        ], [], [
            'contract_agreement' => __('members.contract_agreement')
        ]);
        if ($validation->fails()){
            return ['errors' => $validation->errors()->first()];
        }

        $user->update(['contract_agreement' => 1, 'contract_agreement_date' => now()]);

        return ['message' => __('members.contract_agreement_successfully')];

    }

    public function refreshNotification(Request $request)
    {
        if ($request->ajax()){
            $user = Auth::user();
            $courses = $user->courses()->pluck('course_id')->toArray();
            $notifications = UsersNotification::orderBy('id', 'DESC')
                    ->orWhere('user_id', $user->id)
                    ->orWhereIn('send_to', ['all_users'])
                    ->orWhereIn('course_id', $courses)
                    ->cursor() ?? [];
            $notificationsCount = $notifications->count();
            return view($this->view.'refresh_notification', compact('notifications', 'notificationsCount', 'user'));
        }
    }

    public function markAsRead()
    {
        $user = Auth::user();
        $notifications = UsersNotification::orderBy('id', 'DESC')
            ->where('user_id', $user->id)
            ->orWhere('user_id', null)
            ->cursor();

        foreach ($notifications as $notification) {
            $isRead = $notification->is_read;
            if (is_array($isRead)){
                if (!in_array($user->id, $isRead)){
                    array_push($isRead, $user->id);
                }
            } else {
                $isRead = [$user->id];
            }
            $notification->update(['is_read' => $isRead]);
        }

        return response()->json(['success' => true]);
    }


    /**
     * Check verify code attempt and if it's correct or not
     * @param Request $request
     * @return JsonResponse
     * @throws ValidationException
     */
    public function verifyCode(Request $request)
    {

        $code = implode('',  array_reverse($request->code));

        if (mb_strlen($code) < 4){
            return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.enter_full_code')]);
        }

        $user = User::where('phone', session('provider_phone'))->first()
            ->where('verify_code', $code)->first();

        if (!$user){
            return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.invalid_verification_code')]);
        }

        // Check the number of attempts to check the code
        if ($user->code_verify_attempt >= 6 && strtotime($user->last_attempt_date) == strtotime(date('Y-m-d')) ) {
            return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.verification_tries_exceed')]);
        }

        // Check if date less then today
        if ($user->last_attempt_date && strtotime($user->last_attempt_date) < strtotime(date('Y-m-d'))){
            $user->update(['code_verify_attempt' => 0, 'last_attempt_date' => date('Y-m-d')]);
        }

        // The user entered the wrong code and the number of attempts increased
        if (strtotime($user->last_attempt_date) == strtotime(date('Y-m-d')) && $user->verify_code != $code){
            $user->update(['code_verify_attempt' => $user->code_verify_attempt+1]);
            return $this->checkAttemptCount($request) ?? response()->json(['errors' => __('members.invalid_verification_code')]);
        }

        $user->update(['verify_code' => null]);

        if (session('operation_type') == 'register'){
            $user->update(['email_verified_at' => date('Y-m-d')]);
            session()->put(['country_code' => $user->code, 'is_verified_code' => true]);
            return response()->json([
                'url' => route('provider.completeRegister'),
                'message' => __('members.correct_code')
            ]);
        }
        if (session('operation_type') == 'change_phone'){
            $user->update(['phone' => session('new_phone')]);
            session()->forget(['operation_type', 'provider_phone', 'new_phone', 'is_verified_code' => true]);
            return response()->json([
                'url' => route('provider.profile'),
                'message' => __('members.phone_changed_successfully')
            ]);
        }
    }

    /**
     * check attempt try and generate code
     * @param $user
     * @return array
     */
    private function generateVerifyCode($user)
    {

        /**
         * It's commented for test only and after test
         * delete comment
         */

//        if ($user->attempt_send_code >= 6 && strtotime($user->last_attempt_date) == strtotime(date('Y-m-d')) ) {
//            return ['errors' => __('members.verification_code_exceed')];
//        }

        if ($user->last_attempt_date && strtotime($user->last_attempt_date) < strtotime(date('Y-m-d'))){
            $user->update(['code_verify_attempt' => 0, 'attempt_send_code' => 0,'last_attempt_date' => date('Y-m-d')]);
        }

        $attempt_send_code = $user->attempt_send_code + 1;
        $last_attempt_date = date('Y-m-d');
        $code = rand(1000,9999);

        $phone = $user->phone;
        session()->put(['verify_code' => $code]);

//        try{
//            $message = __('members.sms_message');
//            $message = str_replace('@@', $code, $message);
//            Sms::send('966'.$phone, $message);
//        } catch (Exception $exception){
//            return ['errors' => __('members.sms_error')];
//        }

        /**
         * $data = [
        'user' => $user->name,
        'email' => $user->email,
        'code' => $code,
        ];
        CodeVerificationJob::dispatch($data, 'web.members.emails.verification_email');
         */

        $user->update(
            [
                'verify_code' => $code,
                'attempt_send_code' => $attempt_send_code,
                'last_attempt_date' => $last_attempt_date,
            ]
        );

        return ['errors' => false];
    }


    /**
     * Resend code for register and login
     * @param Request $request
     * @return JsonResponse
     * @throws ValidationException
     */
    public function resendCode(Request $request)
    {
        if ($request->ajax()){
            $user = User::where('phone', session('provider_phone'))->first();

            if ($this->checkAttemptCount($request)){
                return $this->checkAttemptCount($request);
            }

            $generateVerifyCode = $this->generateVerifyCode($user);

            if ($generateVerifyCode['errors']){
                return $this->checkAttemptCount($request) ?? response()->json($generateVerifyCode);
            }

            return response()->json(['success' => true]);
        }
    }

    /**
     * Check user attempts count
     * @param $request
     * @return JsonResponse
     * @throws ValidationException
     */
    public function checkAttemptCount($request)
    {
        if (method_exists($this, 'hasTooManyLoginAttempts') &&
            $this->hasTooManyLoginAttempts($request)) {
            $this->fireLockoutEvent($request);
            return $this->sendLockoutResponse($request);
        }
        $this->incrementLoginAttempts($request);

    }

    /**
     * Override parent sendLockoutResponse to handle it
     * to return json response
     * @param Request $request
     * @return JsonResponse
     */
    protected function sendLockoutResponse(Request $request)
    {
        $seconds = $this->limiter()->availableIn(
            $this->throttleKey($request)
        );
        return response()->json(['errors' => Lang::get('auth.throttle', [
            'seconds' => $seconds,
            'minutes' => ceil($seconds / 60),
        ])]);
    }

    /**
     * Logout for all users
     * @return RedirectResponse
     */
    public function logout()
    {
        Auth::logout();
        session()->forget('http_referer');
        return redirect()->route('home');
    }
}
