<?php


namespace App\Http\Controllers\Web;

use App\Hall;
use App\Helpers\QrCode;
use App\Jobs\SendBillJob;
use App\Payment;
use App\Reservation;
use App\UsersNotification;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MyFatoorah
{

    private static $view = 'web.my_fatoorah.';
    private static $apiKey;
    private static $apiURL;


    public function __construct()
    {
        self::$apiKey = env('MY_FATOORA_TOKEN');
        self::$apiURL = env('MY_FATOORA_URL');
    }

    public function pay(Request $request, Reservation $reservation)
    {

        $validator = \Illuminate\Support\Facades\Validator::make($request->all(),
            ['pay_way' => 'required|numeric|in:6,14,2,11']
            ,
            [],
            ['pay_way' => __('reservations.pay_way')]);

        if ($validator->fails()){
            return response()->json(['errors' => $validator->errors()->first()]);
        }

        $hall = Hall::where('id', $reservation->hall_id)->first();
        $userSelectedDays = $reservation->re_days;
        $availableDates = is_string($hall->available_dates) ? json_decode($hall->available_dates, true) : $hall->available_dates;


        // Check if reserved dates available in hall dates
        foreach ($userSelectedDays['days'] as $userSelectedDay) {
            if (!in_array($userSelectedDay, $availableDates['dates'])){
                $error = str_replace('@@', $userSelectedDay,__('reservations.re_date_no_available'));
                return response()->json(['errors' => $error]);
            }
        }


        // Get dates diff between reserved dates and hall dates
        $remainingDates = array_values(array_diff($availableDates['dates'], $userSelectedDays['days']));

        // Used in completePay function to update available dates
        session()->put(['remaining_dates' => $remainingDates]);

        $user = Auth::user();

        $token = bcrypt(rand(11111,9999).time());
        $check = $this->checkApi($token, $request, $reservation);
        if (isset($check['success']) && $check['success'] == true){
            Payment::create([
                'meta' => ['payment_link' => $check['paymentLink']],
                'token' => $token,
                'user_id' => $user->id
            ]);
            return response()->json(['url' => $check['paymentLink']]);
        } else {
            return redirect()->route('home')->with('alert_pop_up', $check);
        }
    }

    private function checkApi($token, $request, $reservation){
        $user = Auth::user();
        $username= $user->name;
        $phone = $user->phone;
        $email = $user->email;
        $hall = Hall::where('halls.id', $reservation->hall_id)
            ->join('hall_descriptions as hd', 'hd.hall_id', 'halls.id')
            ->where('language_id', currentLanguage()->id)
            ->select(['hd.name'])->first();

        $invoiceItems[] = [
            'ItemName'  => $hall->name,
            'Quantity'  => 1,
            'UnitPrice' => round($reservation->total+$reservation->added_value_amount),
        ];

        $InvoiceValue = round($reservation->total+$reservation->added_value_amount);

        $paymentMethodId = $request->pay_way;
        $callBackUrl = route('callback'); //route('callback');
        $postFields = [
            'CustomerName' => $username,
            'DisplayCurrencyIso' => 'SAR',
            'InvoiceAmount' => '000',
            'CurrencyIso' => 'SAR',
            'CustomerMobile' => $phone,
            'CustomerEmail' => $email,
            'paymentMethodId' => $paymentMethodId,
            'InvoiceValue'    => $InvoiceValue,
            'InvoiceItems'    => $invoiceItems,
            'UserDefinedField' => json_encode(['token' => $token, 'reservation_id' => $reservation->id]),
            'CallBackUrl'     => $callBackUrl,
            'ApiCustomFields' => 'token='.$token,
        ];

        $data = $this->executePayment(self::$apiURL, self::$apiKey, $postFields);

        // If errors found while execute payment
        if (is_array($data) && array_key_exists('errors', $data) && $data['errors'] != ''){
            return $data['errors'];
        }

        $paymentId = $data->InvoiceId;
        $paymentLink = $data->PaymentURL;
        return ['$paymentId' => $paymentId, 'paymentLink' => $paymentLink, 'success' => true];
    }

    private function initiatePayment($apiURL, $apiKey, $postFields) {

        $json = $this->callAPI("$apiURL/v2/InitiatePayment", $apiKey, $postFields);
        return $json->Data->PaymentMethods;
    }

    private function executePayment($apiURL, $apiKey, $postFields) {

        $json = $this->callAPI("$apiURL/v2/ExecutePayment", $apiKey, $postFields);
        // If errors found in call api
        if (is_array($json) && array_key_exists('errors', $json) && $json['errors'] != ''){
            return $json;
        }
        return $json->Data;
    }

    private function callAPI($endpointURL, $apiKey, $postFields = [], $requestType = 'POST') {

        $curl = curl_init($endpointURL);
        curl_setopt_array($curl, array(
            CURLOPT_CUSTOMREQUEST  => $requestType,
            CURLOPT_POSTFIELDS     => json_encode($postFields),
            CURLOPT_HTTPHEADER     => array("Authorization: Bearer $apiKey", 'Content-Type: application/json'),
            CURLOPT_RETURNTRANSFER => true,
        ));

        $response = curl_exec($curl);
        $curlErr  = curl_error($curl);

        curl_close($curl);

        if ($curlErr) {
            //Curl is not working in your server
            return $curlErr;
        }

        $error = $this->handleError($response);
        if ($error) {
            return $error;
        }

        return json_decode($response);
    }

    public function handleError($response) {

        $json = json_decode($response);
        if (isset($json->IsSuccess) && $json->IsSuccess == true) {
            return null;
        }

        //Check for the errors
        if (isset($json->ValidationErrors) || isset($json->FieldsErrors)) {
            $errorsObj = isset($json->ValidationErrors) ? $json->ValidationErrors : $json->FieldsErrors;
            $blogDatas = array_column($errorsObj, 'Error', 'Name');

            $error = implode(', ', array_map(function ($k, $v) {
                return "$k: $v";
            }, array_keys($blogDatas), array_values($blogDatas)));
        } else if (isset($json->Data->ErrorMessage)) {
            $error = $json->Data->ErrorMessage;
        }

        if (empty($error)) {
            $error = (isset($json->Message)) ? $json->Message : (!empty($response) ? $response : 'API key or API URL is not correct');
        }

        return ['errors' => $error];

    }

    public function callback(Request $request)
    {
        $keyId = $request->query('paymentId');
        $KeyType = 'paymentId';

        if (!$keyId){
            return redirect()->route('home')->with('alert_pop_up', 'خطأ في عملية الدفع');
        }

        $postFields = [
            'Key'     => $keyId,
            'KeyType' => $KeyType,
        ];

        $curl = curl_init(self::$apiURL."/v2/getPaymentStatus");
        curl_setopt_array($curl, array(
            CURLOPT_CUSTOMREQUEST  => 'POST',
            CURLOPT_POSTFIELDS     => json_encode($postFields),
            CURLOPT_HTTPHEADER     => array("Authorization: Bearer ".self::$apiKey, 'Content-Type: application/json'),
            CURLOPT_RETURNTRANSFER => true,
        ));

        $response = curl_exec($curl);
        $curlErr  = curl_error($curl);

        curl_close($curl);

        if ($curlErr) {
            return ['errors' => $curlErr];
        }

        $error = $this->handleError($response);
        if ($error) {
            return ['errors' => $error];
        }

        $response = json_decode($response);

//        if ($response->Data->InvoiceTransactions[0]->CustomerServiceCharge < $response->Data->InvoiceValue){
//            return redirect()->route('course.index')->with('alert_pop_up', 'لم تتم عملية الاشتراك، لا يوجد رصيد كافي في حسابك');
//        }

        $payment = null;
        if ($response->IsSuccess == true){
            $token = json_decode($response->Data->UserDefinedField)->token;
            $reservationId = json_decode($response->Data->UserDefinedField)->reservation_id;
            $payment = Payment::where('token', $token);
            if (
                $response->Data->InvoiceStatus != 'Paid' &&
                $response->Data->InvoiceTransactions[0]->TransactionStatus != 'Succss'
            ) {
                $response->payment_url = $payment->first()->meta['payment_link'];
                $payment->update([
                    'meta' => json_encode($response), // ->Data->InvoiceTransactions[0]
                    'invoice_id' => $response->Data->InvoiceId,
                ]);
                return redirect()->route('home')->with('alert_pop_up', __('reservations.payment_error'));
            }

        } else {
            return redirect()->route('home')->with('alert_pop_up', __('reservations.payment_error'));
        }

        $payment = $payment->first();

//        if ($payment && $payment->status){
//            return redirect()->route('home')->with('alert_pop_up', 'غير مسموح لك بهذا الإجراء!');
//        }

        $response->token = $token;
        $response->reservation_id = $reservationId;
//        $response->payment_url = $payment->meta['payment_link'];

//        dd($response);


        $complete = $this->completePay($response, $payment);
        if ($complete){
            return redirect()->route('user.profile')->with('alert_pop_up', __('reservations.reservation_complete'))->with('success', true);
        }


    }

    private function completePay($response, $payment)
    {
        $reservation = Reservation::where('id', $response->reservation_id)->first();
        $user = $reservation->user;
        $provider = $reservation->provider;

        if ($reservation){
            $reservation->update(['re_status' => 'confirmed']);
            $hall = Hall::withDescription($provider->id, $reservation->hall_id);
            $availableDates = session('remaining_dates');
            $hall->update([
                'available_dates' =>
                    [
                        'dates_rang_from' => $hall->available_dates['dates_rang_from'],
                        'dates_rang_to' => $hall->available_dates['dates_rang_to'],
                        'dates' => $availableDates
                    ]
            ]);
        }

        $payment->update([
            'meta' => json_encode($response), // ->Data->InvoiceTransactions[0]
            'invoice_id' => $response->Data->InvoiceId,
            'status' => 1
        ]);

        $userReservation = Reservation::userReservations(null, null, null, null, null, $reservation->id);
//        $services = Service::withDescription($reservation->services);
        $addedServices = [];
        $hallAddedServices = is_string($userReservation->hall_added_service) ? json_decode($userReservation->hall_added_service) : $userReservation->hall_added_service;
//        foreach ($hallAddedServices as $key => $hallAddedService) {
//            if ($reservation->added_services[$key] == $hallAddedService->id){
//                $addedServices[] = $hallAddedService;
//            }
//        }

        $addedValue = array_key_exists('added_value', settings()) ? settings('added_value') : 15;

        /**  Send bill foreach user and provider */

        $seller = settings('website_name');
        $invoiceDate = $reservation->created_at;
        $invoiceTotalAmount = $reservation->net_total;
        $addedValue = settings('added_value');
        $invoiceTaxAmount = ($invoiceTotalAmount*$addedValue)/100;

        $qrCode = QrCode::generate($seller, $invoiceDate, $invoiceTotalAmount, $invoiceTaxAmount);
        try{
            $emails = [$userReservation->provider_email, $userReservation->client_email];
            foreach ($emails as $email) {
                SendBillJob::dispatch([
                    'services' => $reservation->services ?? [],
                    'added_service' => $reservation->added_services ?? [], // $addedServices
                    'added_value' => $addedValue,
                    'reservation'=> $userReservation,
                    'email'=> $email,
                    'qr_code'=> $qrCode['qr_code'],
                ], 'web.reservations.emails.send_bill');
            }

        } catch (\Exception $e){
//            dd($e->getMessage());
        }

        $title = __('halls.confirm_hall_reservation') . "($hall->hall_name)";
        $message = __('halls.hall_reservation_message');
        $messageForUser = str_replace_array('@@', [$user->name, $hall->hall_name], $message);
        $messageForProvider = str_replace_array('@@', [$provider->name, $hall->hall_name], $message);

        $userNotification = UsersNotification::create([
            'title' => $title,
            'message' => $messageForUser,
            'user_id' => $user->id,
            'notification_type' => 'success',
            'page_link' => route('reservation.reservationDetails',$reservation->id),

        ]);
        $providerNotification = UsersNotification::create([
            'title' => $title,
            'message' => $messageForProvider,
            'user_id' => $provider->id,
            'notification_type' => 'success',
            'page_link' => route('reservation.reservationDetails',$reservation->id),
        ]);

        $icon = asset('assets/web/images/icons/profile/noti-success.svg');
        $now = Carbon::now()->setTimezone('Asia/riyadh');
        $notificationTime = Carbon::make($userNotification->created_at)->setTimezone('Asia/riyadh');
        $notificationTime = $now->diffInMinutes($notificationTime);
        $data = [
            'reservation_done' => 'done',
            'reservation_route' => route('reservation.reservationDetails',$reservation->id),
            'reservation_image' => asset(json_decode($userReservation->images, true)[0]),
            'hall_name' => $hall->hall_name,
            're_no' => "$userReservation->re_no",
            're_date_from' => date('d/m/Y', strtotime($userReservation->re_date_from)),
            'person_count' => "$userReservation->person_count",
            'total' => "$userReservation->total",
            'user_photo' => asset($user->photo ?? 'assets/web/images/avatar.png'),
            'notification_time' => $notificationTime . __('halls.minutes') . __('halls.ago'),
            'notification_date' => date('M Y d', strtotime($reservation->created_at)),
            'notification_icon' => $icon,
            'notification_class' => '',
            'reservation_container_id' => '#pills-completed',
            'badge' => 'badge_completed',
            'badge_title' => __('reservations.confirmed'),
        ];
        $userData = array_merge([
            'name' => $user->name,
            'notification_title' => $title,
            'notification_message' => $messageForUser,
            'notification_id' => (string) $userNotification->id
        ], $data);
        $providerData = array_merge([
            'name' => $provider->name,
            'notification_title' => $title,
            'notification_message' => $messageForProvider,
            'notification_id' => (string) $providerNotification->id
        ], $data);

        if (is_array($user->device_token) && isset($user->device_token['token'])){
            $notificationForUser = new SendNotification($title, $messageForUser, $icon, $userData, route('user.profile'));
            $user->notify($notificationForUser);
        }

        if (is_array($user->device_token) && isset($user->device_token['token'])){
            $notificationForProvider = new SendNotification($title, $messageForProvider, $icon, $providerData, route('provider.profile'));
            $provider->notify($notificationForProvider);
        }
        return true;
    }


    /**
     * Specific payment way
     * @param Request $request
     * @param Reservation $reservation
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse
     */
//    public function payForm(){
//
//        if (!session('pay_way') && (!session('course') || !session('orders'))){
//            return redirect()->route('home');
//        }
//
////        $ipPostFields = ['InvoiceAmount' => '000', 'CurrencyIso' => 'SAR'];
////
////        $paymentMethods = $this->initiatePayment(self::$apiURL, self::$apiKey, $ipPostFields);
////
////        $ways = collect();
////        // [VISA/MASTER' => 2, 'Sadad' => 4, 'MADA' => 6, 'Apple Pay' => 11, 'STC Pay' => 14]
////        // [2,4,6,11,14]
////
////        foreach($paymentMethods as $paymentMethod){
////            if(in_array($paymentMethod->PaymentMethodId, [2,6])){
////                $ways->push([
////                    'PaymentMethodId'=>$paymentMethod->PaymentMethodId,
////                    'PaymentMethodAr'=>$paymentMethod->PaymentMethodAr,
////                    'PaymentMethodEn'=>$paymentMethod->PaymentMethodAr
////                ]);
////            }
////        }
//
//        $url = session('course') ? route('pay') : route('eyeStore.productPay');
//
//        return redirect($url);
//
////        return view(self::$view.'pay',compact('ways', 'url'));
//    }

}
