<?php

namespace App\Http\Controllers\Web;

use App\Category;
use App\City;
use App\Hall;
use App\Occasion;
use App\Http\Controllers\Controller;
use App\Service;
use App\Table;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Route;


class HallController extends Controller
{

    protected $view = 'web.halls.';

    public function index(Request $request)
    {
        $title = __('halls.head');
        $filter = $this->filter($request);
        $occasions = $filter['occasions'];
        $categories = $filter['categories'];
        $cities = $filter['cities'];
        $halls = $filter['halls'];

        if (request()->ajax()){
            return view($this->view.'load_more', get_defined_vars());
//            return view($this->view.'next_hall', compact('halls'));
        }

        return view($this->view.'index', get_defined_vars());
    }

    public function show(Hall $hall)
    {
        $title = __('halls.hall_details');
        $services = $hall->allow_services ? Service::withDescription(null, $hall->reservation_type) : [];
        $addedServices = is_string($hall->added_services)? json_decode($hall->added_services, true) : $hall->added_services;
        $occasions = Occasion::withDescription($hall->occasions);
        $tables = is_array($hall->table_ids) && count($hall->table_ids) ? Table::withDescription($hall->table_ids) : [];

        $evaluates = DB::table('evaluates')
            ->whereNull('evaluates.deleted_at')
            ->where('evaluates.status', '=', 1)
            ->where('evaluates.hall_id', '=',  $hall->id)
            ->join('users', 'users.id', '=', 'evaluates.user_id')
            ->select(['evaluates.comment','evaluates.evaluate_average', 'users.name as user_name', 'users.photo as user_photo'])
            ->cursor();

        $hall = $this->getHalls()->where('halls.id', $hall->id)->first();

        /**
         * Check if whether region or city or sector not deleted
         */
        if (!$hall){
            return redirect()->route('home');
        }

        $hall->update(['views_counts' => $hall->views_counts+1]);

        return view($this->view.'show', get_defined_vars());
    }

    public function offerHall(Request $request)
    {
        $title = __('halls.special_offer');
        $filter = $this->filter($request);
        $occasions = $filter['occasions'];
        $categories = $filter['categories'];
        $cities = $filter['cities'];
        $halls = $filter['halls'];



        if (request()->ajax()){
            return view($this->view.'load_more', get_defined_vars());
        }

        return view($this->view.'offer_halls', get_defined_vars());
    }

    private function getHalls()
    {
        $languageId = currentLanguage()->id;

        $halls = Hall::
//        orderByDesc('halls.id')
            join('hall_descriptions as hd', 'hd.hall_id', 'halls.id')
            ->where('hd.language_id', $languageId)
            ->join('categories', 'categories.id', 'halls.category_id')
            ->join('category_descriptions as cat_desc', 'cat_desc.category_id', 'categories.id')
            ->where('cat_desc.language_id', $languageId)
            ->join('regions', 'regions.id', 'halls.region_id')
//            ->where('regions.deleted_at', null)
            ->join('region_descriptions as region_desc', 'region_desc.region_id', 'regions.id')
            ->where('region_desc.language_id', $languageId)
            ->join('cities', 'cities.id', 'halls.city_id')
//            ->where('cities.deleted_at', null)
            ->join('city_descriptions as city_desc', 'city_desc.city_id', 'cities.id')
            ->where('city_desc.language_id', $languageId)
            ->join('sectors', 'sectors.id', 'halls.sector_id')
//            ->where('sectors.deleted_at', null)
            ->join('sector_descriptions as sector_desc', 'sector_desc.sector_id', 'sectors.id')
            ->where('sector_desc.language_id', $languageId)
            ->join('users', 'halls.user_id', 'users.id')
            ->select(
                [
                    'halls.*',
                    'hd.name as hall_name',
                    'hd.description as hall_description',
                    'hd.reservation_condition',
                    'hd.features',
                    'cat_desc.name as category_name',
                    'users.name as provider_name',
                    'users.photo as provider_photo',
                    'users.logo as provider_logo',
                    'region_desc.name as region_name',
                    'city_desc.name as city_name',
                    'sector_desc.name as sector_name',
                ]
            )
//            ->selectSub("SELECT ROUND(AVG(evaluate_average)) FROM evaluates WHERE status = 1 AND hall_id = halls.id", 'rate')
            ->selectSub("SELECT COUNT(hall_id) FROM evaluates WHERE status = 1 AND `deleted_at` IS NULL AND hall_id = halls.id", 'evaluate_counts')
            ->selectSub("SELECT COUNT(hall_id) FROM reservations  WHERE `re_status` = 'finished' AND `deleted_at` IS NULL AND hall_id = halls.id", 'reservation_counts');

        return $halls;
    }

    private function filter($request){
        $name = $request->query('hall_name') ?? $request->hall_name;
        $hallFiltration = $request->query('hall_filtration') ?? null;
        $city_id = $request->query('city_id') ?? $request->city_id;
        $occasion_id = $request->query('occasion_id') ?? $request->occasion_id;
        $category_id = $request->query('category_id') ?? $request->category_id;
        $rate = $request->query('rate') ?? $request->rate;
        $re_date_from = $request->query('re_date_from') ? date('Y-m-d', strtotime($request->query('re_date_from'))) : null;
        $re_date_to = $request->query('re_date_to') ? date('Y-m-d', strtotime($request->query('re_date_to'))) : $re_date_from;
        $skip = $request->skip ?? 0;
        $hallOffer = Route::currentRouteName();

        $occasions = Occasion::withDescription();
        $categories = Category::withDescription();
        $cities = City::whereHas('halls')
            ->join('city_descriptions as cd', 'cd.city_id', '=', 'cities.id')
            ->where('cd.language_id', '=', currentLanguage()->id)
            ->select([
                'cities.*',
                'cd.name',
            ])->cursor();

        $halls = $this->getHalls();


        if ($hallFiltration){

            if ($hallFiltration == 'highest_viewing'){
                $halls->where('halls.views_counts', '!=', null)->orderByDesc('halls.views_counts');
            }
            if ($hallFiltration == 'highest_rated'){
                $halls->where('halls.rate', '!=', null)->orderByDesc('halls.rate');
            }
            if ($hallFiltration == 'most_requested'){
                $halls->orderByDesc('reservation_counts');
            }

        }

        if ($occasion_id != 'all' && $occasion_id) {
            $halls->whereJsonContains('halls.occasions', $occasion_id);
        }

        if ($city_id != 'all' && $city_id) {
            $halls->where('halls.city_id', $city_id);
        }

        if ($category_id != 'all' && $category_id) {
            $halls->where('halls.category_id', $category_id);
        }

        if ($rate != 'all' && $rate){
            $halls->where('halls.rate', $rate);
        }

        if ($name){
            $halls->where('hd.name', 'LIKE', '%'.$name.'%');
        }

        if($hallOffer == 'hall.offerHall'){
            $halls->where('halls.is_offer', 1);
        }

        $halls_ids = [];
        if ($re_date_from && $re_date_to && ($re_date_to >= $re_date_from)){
//            $halls->where('halls.available_dates', 'LIKE', '%'.$re_date_from.'%');

            // Here generate dates from => to
            $dates = [];
//            $from = Carbon::make($re_date_from)->addDays(-1);
            $from = Carbon::make($re_date_from)->setTimezone('Asia/Riyadh');
            $to = Carbon::make($re_date_to);
            $diff = $to->diff($from)->days;
            if ($diff){
                while($to->diff($from)->days){
                    $from = $from->addDay();
                    $dates[] = $from->format('Y-m-d');
                }
            } else {
                $dates[] = $from->format('Y-m-d');
                session()->put(['dates_from_search_filter' => $from->format('Y-m-d')]);
            }

            $checkHalls = $halls->cursor();
            foreach ($checkHalls as $key => $hall) {

                // Check if dates found in available dates or not
                $isDateFound = array_filter($hall->availableDate['dates'], function($value) use ($dates){
                    return in_array($value, $dates);
                });

                // Here is date found it means in available dates
                // Push hall is to halls_id array
                if (count($isDateFound)){
                    $halls_ids[] = $hall->id;
                }
//                $hallFrom = date('Y-m-d', strtotime($hall->availableDate[0]));
//                $hallTo = date('Y-m-d', strtotime(array_last($hall->availableDate)));
//                if (!($re_date_to >= $re_date_from && $re_date_from <= $hallTo) || !(($re_date_from <= $hallFrom && $re_date_to >= $hallFrom) || ($re_date_to >= $hallTo && $re_date_to >= $hallFrom))){
//                    unset($halls[$key]);
//                }
            }
        }

        if (count($halls_ids)){
//            $halls = $halls->whereIn('halls.id', $halls_ids)->simplePaginate(9);
            $halls = $halls->whereIn('halls.id', $halls_ids)->skip($skip)->limit(9)->cursor(); // paginate(9)
        } elseif(!count($halls_ids) && ($re_date_from && $re_date_to)){
            $halls = [];
        } elseif(!count($halls_ids) && !$re_date_from && !$re_date_to) {
            session()->forget(['dates_from_search_filter']);
//            $halls = $halls->simplePaginate(9);
            $halls = $halls->skip($skip)->limit(9)->cursor(); // paginate(9)

        }

        return ['halls' => $halls, 'occasions' => $occasions, 'categories' => $categories, 'cities' => $cities];
    }
}
