<?php

namespace App\Http\Controllers\Admin;

use App\Http\Requests\UsersNotificationRequest;
use App\User;
use App\UsersNotification;
use Exception;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;

class UsersNotificationController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/usersNotifications';
    /** View folder */
    private $viewDirectory = 'admin.users_notifications.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('usersNotifications.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index',get_defined_vars());
    }

    /**
     * return data of the trainingBags.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = UsersNotification::orderBy('id', 'desc');

        if ($request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        if ($request->title) {
            $query->where('title', 'LIKE', '%'.$request->title.'%');
        }


        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $usersNotifications = $query->paginate(100);

        return view($this->viewDirectory.'grid',get_defined_vars());
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('usersNotifications.store');
        $head = metaFields('usersNotifications', 'add_new', getCurrentLocale()) ?? __('usersNotifications.new');
        $title = $head;
        $users = [];
        $role = null;

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param UsersNotificationRequest $request
     * @return JsonResponse
     */
    public function store(UsersNotificationRequest $request) :JsonResponse
    {
        $data = $request->all();

        if (in_array($request->user_id, ['all_user', 'all_provider'])){
            $data['send_to'] = $request->user_id;
            $data['user_id'] = null;
        } else {
            $data['send_to'] = null;
        }

        UsersNotification::create($data);
        session()->flash('message', __('dashboard.saveDone'));
        return  response()->json(['url' => route('usersNotifications.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param UsersNotification $usersNotification
     * @return View
     */
    public function edit(UsersNotification $usersNotification) :View
    {
        $action= route('usersNotifications.update', $usersNotification->id);
        $head = metaFields('trainingBags', 'edit', getCurrentLocale()) ?? __('usersNotifications.edit');
        $title = $head;
        $users = null;
        if ($usersNotification->send_to == 'all_user' || (($usersNotification->user_id && $usersNotification->user->role == 'user'))){
            $users = User::where('role', 'user')->cursor();
            $role = 'all_user';
        }
        if ($usersNotification->send_to == 'all_provider' || (($usersNotification->user_id && $usersNotification->user->role == 'provider'))){
            $users = User::where('role', 'provider')->cursor();
            $role = 'all_provider';
        }

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     *
     * @param UsersNotificationRequest $request
     * @param UsersNotification $usersNotification
     * @return JsonResponse
     */
    public function update(UsersNotificationRequest $request, UsersNotification $usersNotification) :JsonResponse
    {
        $data = $request->all();

        if (in_array($request->user_id, ['all_user', 'all_provider'])){
            $data['send_to'] = $request->user_id;
            $data['user_id'] = null;
        } else {
            $data['send_to'] = null;
        }

        $usersNotification->update($data);
        session()->flash('message', __('dashboard.saveDone'));
        return  response()->json(['url' => route('usersNotifications.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }


    public function getUsers(Request $request)
    {
        if ($request->ajax()){
            $role  = $request->role == 'all_user' ? 'user' : 'provider';
            $users = User::where('role', $role)->cursor();
            return ['users' => $users, 'role' => $request->role, 'label' => __('usersNotifications.'.$role)];
        }
    }



    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $trainingBag = UsersNotification::withTrashed()->find($id);
        if ($trainingBag) {
            if ($trainingBag->deleted_at) {
                $trainingBag->forceDelete();
            } else {
                $trainingBag->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several trainingBags by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            UsersNotification::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            UsersNotification::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $trainingBag = UsersNotification::withTrashed()->find($id);

        if ($trainingBag){
            $trainingBag->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several trainingBags by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $trainingBags = UsersNotification::whereIn('id', $request->ids)->onlyTrashed();
        if ($trainingBags){
            $trainingBags->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
