<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\UserRequest;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;
use App\User;
use App\Module;

class UserController extends Controller
{
    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/users';
    /** View folder */
    private $viewDirectory = 'admin.users.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('users.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the users.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = User::orderBy('role', 'ASC')
            ->orderBy('name', 'ASC');

        if (Auth::user()->role == 'admin' && Auth::user()->admin_token){
            $query->where('admin_token', null)->orWhere('id', Auth::id())->whereIn('role', ['admin', 'sub_admin']);
        } elseif(Auth::user()->role == 'admin' && Auth::user()->admin_token == null){
            $query->where('admin_token', null)->whereIn('role', ['admin', 'sub_admin']);
        } elseif(Auth::user()->role == 'sub_admin'){
            $query->where('admin_token', null)->where('role', 'sub_admin');
        }


        if ($request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('name', 'LIKE', '%'.$request->name.'%');
        }

        if ($request->email) {
            $query->where('email', 'LIKE', '%'.$request->email.'%');
        }

        if ($request->role) {
            $query->where('role', strtolower($request->role) );
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $users= $query->paginate(100);

        return view($this->viewDirectory.'grid',get_defined_vars());
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('users.store');
        $head = metaFields('users', 'add_new', getCurrentLocale()) ?? __('users.new');
        $title = $head;

        $modules= Module::where('deleted_at', null)->cursor();
        $user_modules= old('modules',  []);

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param UserRequest $request
     * @return RedirectResponse|Redirector
     */
    public function store(UserRequest $request) :RedirectResponse
    {
        $data = $request->all();

        $data['password'] = bcrypt($request->password);

        $user= User::create($data);

        $user->modules()->sync($request->modules);

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  User  $user
     * @return View
     */
    public function edit(User $user): View
    {
        $action= route('users.update', $user->id);
        $head = metaFields('users', 'edit', getCurrentLocale()) ?? __('users.edit');
        $title = $head;

        $user_modules= old('modules',  $user->modules()->pluck('modules.id')->toArray() ?? []);

        if ('admin' != $user->role) {
            $modules= Module::where('deleted_at', null)->cursor();
        } else {
            $modules= [];
        }

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  UserRequest  $request
     * @param  User  $user
     * @return RedirectResponse|Redirector
     */
    public function update(UserRequest $request, User $user) :RedirectResponse
    {
        // dd($request->all());
        $data= $request->all();

        if($data['password']) {
            $data['password']= bcrypt( $data['password'] );
        } else {
            unset($data['password']);
        }
        unset($data['email']); // cannot change the email
        $user->update($data);
        // dd($user->toArray());
        // Insert modules
        if ($request->modules){
            $user->modules()->sync($request->modules);
        }

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $user = User::withTrashed()->find($id);
        if ($user) {
            if ($user->deleted_at) {
                $user->forceDelete();
            } else {
                $user->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            User::onlyTrashed()->whereIn('id', $ids)->where('role', '!=', 'admin')->forceDelete();
        } else {
            User::whereIn('id', $ids)->where('role', '!=', 'admin')->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $user = User::withTrashed()->find($id);

        if ($user){
            $user->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $user = User::whereIn('id', $request->ids)->onlyTrashed();
        if ($user){
            $user->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
