<?php

namespace App\Http\Controllers\Admin;

use App\Table;
use Exception;
use App\TableDescription;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Requests\TableRequest;
use Illuminate\Routing\Redirector;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;

class TableController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/tables';
    /** View folder */
    private $viewDirectory = 'admin.tables.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('tables.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the tables.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Table::orderByDesc('tables.id')
            ->join('table_descriptions AS tDesc', 'tables.id', 'tDesc.table_id')
            ->join('languages', 'languages.id', 'tDesc.language_id')
            ->where('local', getCurrentLocale())
            ->select(['tables.*', 'tDesc.name']);

        if ($request->date_from) {
            $query->whereDate('tables.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('tables.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('tDesc.name', 'LIKE', '%'.$request->name.'%');
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }


        $tables = $query->paginate(100);

        return view($this->viewDirectory.'grid', get_defined_vars());
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('tables.store');
        $head = metaFields('tables', 'add_new', getCurrentLocale()) ?? __('tables.new');
        $title = $head;

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param TableRequest $request
     * @return RedirectResponse|Redirector
     */
    public function store(TableRequest $request) :RedirectResponse
    {
        $table = Table::create($request->all());
        // Insert Description
        $this->saveData( $request, $table->id );

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param Table $table
     * @return View
     */
    public function edit(Table $table) :View
    {
        $action= route('tables.update', $table->id);
        $head = metaFields('tables', 'edit', getCurrentLocale()) ?? __('tables.edit');
        $title = $head;

        $query = TableDescription::where('table_id', $table->id)
            ->join('languages', 'languages.id', 'table_descriptions.language_id')
            ->select(['table_descriptions.*', 'languages.local']);

        $tableDescription= $query->get();

        foreach ($tableDescription as $row) {
            $table[$row->local]= $row;
        }

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     *
     * @param TableRequest $request
     * @param Table $table
     * @return RedirectResponse|Redirector
     */
    public function update(TableRequest $request, Table $table) :RedirectResponse
    {
        // Update image if request has it
        $table->update($request->all());

        // Delete old description
        TableDescription::where('table_id', $table->id)->delete();

        // Insert new description
        $this->saveData( $request, $table->id );

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Handle Save form data
     *
     * @param TableRequest $request
     * @param int $table_id
     * @return void
     */

    private function saveData(TableRequest $request, int $table_id ):void
    {
        $requestData= $request->all();
        foreach (languages() as $lang) {
            $data=[
                'name'=> $requestData['name_'.$lang->local],
                'table_id'=> $table_id,
                'language_id'=> $lang->id,
            ];
            TableDescription::create($data);
        }

    }


    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $table = Table::withTrashed()->find($id);
        if ($table) {
            if ($table->deleted_at) {
                $table->forceDelete();
            } else {
                $table->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several tables by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            Table::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            Table::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $table = Table::withTrashed()->find($id);

        if ($table){
            $table->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several tables by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $table = Table::whereIn('id', $request->ids)->onlyTrashed();
        if ($table){
            $table->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
