<?php

namespace App\Http\Controllers\Admin;

/** Ajax Request */

use App\Category;
use App\Service;
use Exception;
use Illuminate\View\View;
use App\ServiceDescription;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Redirector;
use Illuminate\Http\RedirectResponse;
use App\Http\Controllers\Controller;
use App\Http\Requests\ServiceRequest; // for Validation

class ServiceController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/services';
    /** View folder */
    private $viewDirectory = 'admin.services.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the services.
     *
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('services.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the services.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Service::latest()
        ->join('service_descriptions AS sd', 'services.id', 'sd.service_id')
        ->join('languages', 'languages.id', 'sd.language_id')
        ->where('local', getCurrentLocale())
        ->select(['sd.name', 'services.*']);

        if ($request->date_from) {
            $query->whereDate('services.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('services.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('sd.name', 'LIKE', '%'.$request->name.'%');
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }
        $services= $query->paginate(100);

        return view($this->viewDirectory.'grid',get_defined_vars());
    }

    /**
     * Show the form for creating a new service.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('services.store');
        $head = metaFields('services', 'add_new', getCurrentLocale()) ?? __('services.new');
        $title = $head;
//        $categories = Category::withDescription();
        $local = getCurrentLocale();
        return view( $this->viewDirectory.'form', get_defined_vars() );
    }

    /**
     * Store a newly created service in storage.
     * @param ServiceRequest $request
     * @return RedirectResponse|Redirector
     */
    public function store(ServiceRequest $request) :RedirectResponse
    {
        $service = Service::create($request->all());
        $this->saveData( $request, $service->id );
        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Show the form for editing the specified service.
     *
     * @param  Service  $service
     * @return View
     */
    public function edit(Service $service) :View
    {
        $action= route('services.update', $service->id);
        $head = metaFields('services', 'edit', getCurrentLocale()) ?? __('services.edit');
        $title = $head;

        $query = ServiceDescription::where('service_id', $service->id)
        ->join('languages', 'languages.id', 'service_descriptions.language_id')
        ->select(['service_descriptions.*', 'languages.local']);

        $serviceDescription= $query->get();

        foreach ($serviceDescription as $row) {
            $service[$row->local]= $row;
        }

//        $categories = Category::withDescription();
        $local = getCurrentLocale();

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified service in storage.
     * @param ServiceRequest $request
     * @param Service $service
     * @return RedirectResponse|Redirector
     */
    public function update(ServiceRequest $request, Service $service) :RedirectResponse
    {
        $service->update($request->all());
        ServiceDescription::where('service_id', $service->id)->delete();
        $this->saveData( $request, $service->id );
        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Handle Save form data
     *
     * @param ServiceRequest $request
     * @param int $service_id
     * @return void
     */

     private function saveData(ServiceRequest $request, int $service_id ) :void
     {

        $requestData= $request->all();

        foreach (languages() as $lang) {
            $data=[
                'service_id'=> $service_id,
                'language_id'=> $lang->id,
                'name'=> $requestData['name_'.$lang->local],
                'description'=> $requestData['description_'.$lang->local],
            ];

            ServiceDescription::create($data);
        }

     }

    /**
     * Remove the specified service from storage.
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $service= Service::withTrashed()->find($id);
        if ($service) {
            if ($service->deleted_at) {
                $service->forceDelete();
            } else {
                $service->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several services by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;

        if ($request->force) {
            Service::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            Service::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified service from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $service= Service::withTrashed()->find($id);
        if ($service) {
            $service->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);

    }

    /**
     * Restore several services by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        Service::whereIn('id', $ids)->onlyTrashed()->restore();
        return response()->json(['message'=> __('dashboard.saveDone')]);
    }
}
