<?php

namespace App\Http\Controllers\Admin;

/** Ajax Request */

use App\Helpers\QrCode;
use App\Reservation;

use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;

class ReservationController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/reservations';
    /** View folder */
    private $viewDirectory = 'admin.reservations.';

    /**
     * Display a listing of the reservations.
     *
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('reservations.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the reservations.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Reservation::orderBy('id', 'desc')
        ->whereNotNull('reservations.re_status');

        if ($request->date_from) {
            $query->whereDate('reservations.updated_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('reservations.updated_at', '<=', $request->date_to);
        }

        if ($request->re_status) {
            $query->where('reservations.re_status', $request->re_status);
        }

        if ($request->re_way) {
            $query->where('reservations.re_way', $request->re_way);
        }

        if ($request->re_no) {
            $query->where('reservations.re_no', 'LIKE', '%'.$request->re_no.'%');
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }
        $reservations= $query->paginate(100);

        return view($this->viewDirectory.'grid',get_defined_vars());
    }


    /**
     * Show the form for editing the specified service.
     *
     * @param  Reservation  $reservation
     * @return View
     */
    public function edit(Reservation $reservation)
    {
        $action= route('reservations.update', $reservation->id);
        $head = metaFields('reservations', 'edit', getCurrentLocale()) ?? __('reservations.edit');
        $title = $head;

        if ($reservation->re_status == null){
            return redirect()->route('reservations.index');
        }

        $reservation = Reservation::where('reservations.id', $reservation->id)
        ->join('users', 'reservations.user_id', 'users.id')
        ->join('users as providers', 'reservations.provider_id', 'providers.id')
        ->join('halls', 'halls.id', 'reservations.hall_id')
        ->join('hall_descriptions as hd', 'halls.id', 'hd.hall_id')
        ->where('hd.language_id', currentLanguage()->id)
            ->select([
                'reservations.*',
                'hd.name as hall_name',
                'providers.name as provider_name',
                'users.name as user_name',
                'users.phone as user_phone',
            ])->first();

        $local = getCurrentLocale();

        $servicesAndAddedServices = $this->getReservationBillDetails($reservation)['details'];

        $totalBeforeAddedValue = $reservation->total; // Total after discount and before added value
        $addedValueAmount = $reservation->added_value_amount;
        $totalAfterAddedValue = $totalBeforeAddedValue+$addedValueAmount;
        $sitePercentageValue = ($totalBeforeAddedValue*settings('website_percentage'))/100;


        $reservationDays = $reservation['re_days'];

        $seller = settings('website_name');
        $invoiceDate = $reservation->created_at;
        $invoiceTotalAmount = $reservation->net_total;
        $addedValue = settings('added_value');
        $invoiceTaxAmount = ($invoiceTotalAmount*$addedValue)/100;

        $qrCode = QrCode::generate($seller, $invoiceDate, $invoiceTotalAmount, $invoiceTaxAmount);

        return view( $this->viewDirectory.'form', get_defined_vars() );
    }

    private function getReservationBillDetails($reservation){
        $services = is_string($reservation->services) ? json_decode($reservation->services) : ($reservation->services ?? []);
        $addedServices = is_string($reservation->added_services) ? json_decode($reservation->added_services) : ($reservation->added_services ?? []);
        $reDays = $reservation->re_days;
        $servicesTotal = 0;
        $addedServiceTotal = 0;

        $servicesAndAddedServices = [];
        if (count($services)){
            foreach ($services as $day => $service) {
                foreach ($service as $oneService) {
                    $servicesTotal += $oneService['price']*$oneService['quantity'];
                    $servicesAndAddedServices[$day]['services'][] = $oneService;
                }
            }
        }

        if (count($addedServices)){
            foreach ($addedServices as $day => $addedService) {
                foreach ($addedService as $oneAddedService) {
                    $addedServiceTotal += $oneAddedService['price']*$oneAddedService['quantity'];
                    $servicesAndAddedServices[$day]['added_services'][] = $oneAddedService;

                }
            }
        }


        foreach ($reDays['days'] as $key =>  $reDay) {
            if (count($services) && !key_exists($reDay, $services)){
                $servicesAndAddedServices[$reDay]['services'][] = [
                    "id" => null,
                    "service_type" => $reservation->reservation_type,
                    "image" => null,
                    "price" => 0,
                    "language_id" => 1,
                    "name" => null,
                    "description" => null,
                    "quantity" => 0,
                    "selected_day" => $reDay,
                    "discount_percentage" => $reDays['percentage'][$key],
                ];
            }
            if (count($addedServices) && !key_exists($reDay, $addedServices)){
                $servicesAndAddedServices[$reDay]['added_services'][] = [
                    "id" => null,
                    "service_type" => $reservation->reservation_type,
                    "image" => null,
                    "price" => 0,
                    "language_id" => 1,
                    "name" => null,
                    "description" => null,
                    "quantity" => 0,
                    "selected_day" => $reDay,
                    "discount_percentage" => $reDays['percentage'][$key],
                ];
            }
        }


        ksort($servicesAndAddedServices);

        return ['details' => $servicesAndAddedServices];

    }


    /**
     * Update the specified service in storage.
     * @param Request $request
     * @param \App\Http\Controllers\Admin\Reservation $reservation
     * @return JsonResponse
     */
    public function update(Request $request, Reservation $reservation) :JsonResponse
    {
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('reservations.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

}
