<?php

namespace App\Http\Controllers\Admin;

use App\Category;
use App\Product;
use App\Http\Requests\ProductRequest;
use Exception;
use App\ProductDescription;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;

class ProductController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/products';
    /** View folder */
    private $viewDirectory = 'admin.products.';

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('products.head');
        $request= $request->toArray();
        $categories = Category::withDescription();
        return view($this->viewDirectory.'index',compact('title', 'categories'));
    }

    /**
     * return data of the courses.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Product::latest()
            ->join('product_descriptions AS pd', 'products.id', 'pd.product_id')
            ->where('pd.language_id', currentLanguage()->id)
            ->join('categories', 'categories.id', 'products.category_id')
            ->join('category_descriptions AS cd', 'categories.id', 'cd.category_id')
            ->where('cd.language_id', currentLanguage()->id)
            ->select(['pd.name','cd.name as category_name', 'categories.id as category_id', 'products.*']);

        if ($request->date_from) {
            $query->whereDate('products.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('products.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('pd.name', 'LIKE', '%'.$request->name.'%');
        }

        if ($request->category_id) {
            $query->where('categories.id', $request->category_id);
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $products= $query->paginate(100);

        return view($this->viewDirectory.'grid',compact('products'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('products.store');
        $head = metaFields('products', 'add_new', getCurrentLocale()) ?? __('products.new');
        $categories = Category::withDescription();
        return view( $this->viewDirectory.'form', compact('action', 'head', 'categories'));
    }

    /**
     * Store a newly created resource in storage.
     * @param ProductRequest $request
     * @return JsonResponse
     */
    public function store(ProductRequest $request) :JsonResponse
    {
        $product = Product::create($request->all());
        $this->saveData( $request, $product->id );
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('products.index')]);
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param Product $product
     * @return View
     */
    public function edit(Product $product) :View
    {
        $action= route('products.update', $product->id);
        $head = metaFields('products', 'edit', getCurrentLocale()) ?? __('products.edit');

        $query = ProductDescription::where('product_id', $product->id)
            ->join('languages', 'languages.id', 'product_descriptions.language_id')
            ->select(['product_descriptions.*', 'languages.local']);

        $productDescription= $query->get();

        foreach ($productDescription as $row) {
            $product[$row->local]= $row;
        }
        $categories = Category::withDescription();

        return view( $this->viewDirectory.'form', compact('product', 'action', 'head', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     * @param ProductRequest $request
     * @param Product $product
     * @return JsonResponse
     */
    public function update(ProductRequest $request, Product $product) :JsonResponse
    {
        $data = $request->all();
        $product->update($data);
        // Delete old description
        ProductDescription::where('product_id', $product->id)->delete();

        $this->saveData( $request, $product->id );

        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('products.index')]);
    }


    /**
     * Handle Save form data
     *
     * @param ProductRequest $request
     * @param int $product_id
     * @return void
     */

    private function saveData(ProductRequest $request, int $product_id ):void
    {
        $requestData= $request->all();
        foreach (languages() as $lang) {
            $data=[
                'product_id'=> $product_id,
                'language_id'=> $lang->id,
                'name'=> $requestData['name_'.$lang->local],
                'description'=> $requestData['description_'.$lang->local],
            ];
            ProductDescription::create($data);
        }

    }



    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $product = Product::withTrashed()->find($id);
        if ($product) {
            if ($product->deleted_at) {
                $product->forceDelete();
            } else {
                $product->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several courses by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            Product::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            Product::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $product = Product::withTrashed()->find($id);

        if ($product){
            $product->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several courses by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $products = Product::whereIn('id', $request->ids)->onlyTrashed();
        if ($products){
            $products->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
