<?php

namespace App\Http\Controllers\Admin;

use App\Http\Requests\ProductBlockRequest;
use App\Product;
use App\ProductBlock;
use App\ProductBlockDescription;
use Exception;
use App\Language;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Redirector;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;

class ProductBlockController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/productBlocks';
    /** View folder */
    private $viewDirectory = 'admin.product_blocks.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('productBlocks.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index',compact('title'));
    }

    /**
     * return data of the productBlocks.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = ProductBlock::orderBy('product_blocks.id', 'DESC')
            ->join('product_block_descriptions AS pd', 'product_blocks.id', 'pd.product_block_id')
            ->join('languages', 'languages.id', 'pd.language_id')
            ->where('pd.language_id', currentLanguage()->id)
            ->select(['pd.title', 'product_blocks.*']);

        if ($request->date_from) {
            $query->whereDate('productBlocks.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('productBlocks.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('pd.title', 'LIKE', '%'.$request->name.'%');
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $productBlocks= $query->paginate(10);

        return view($this->viewDirectory.'grid',compact('productBlocks'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create()
    {
        $action= route('productBlocks.store');
        $head = metaFields('productBlocks', 'add_new', getCurrentLocale()) ?? __('productBlocks.new');
        $products = Product::withDescription();
        return view( $this->viewDirectory.'form', compact('products', 'action', 'head') );
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param ProductBlockRequest $request
     * @return RedirectResponse|Redirector
     */
    public function store(ProductBlockRequest $request) :RedirectResponse
    {
        $productBlock = ProductBlock::create($request->all());
        // Insert Description
        $this->saveData( $request, $productBlock->id );

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param ProductBlock $productBlock
     * @return View
     */
    public function edit(ProductBlock $productBlock) :View
    {
        $action= route('productBlocks.update', $productBlock->id);
        $head = metaFields('productBlocks', 'edit', getCurrentLocale()) ?? __('productBlocks.edit');

        $query = ProductBlockDescription::where('product_block_id', $productBlock->id)
            ->join('languages', 'languages.id', 'product_block_descriptions.language_id')
            ->select(['product_block_descriptions.*', 'languages.local']);

        $productBlockDescription= $query->get();

        foreach ($productBlockDescription as $row) {
            $productBlock[$row->local]= $row;
        }

        $products = Product::withDescription();

        return view( $this->viewDirectory.'form', compact('productBlock', 'products','action', 'head') );
    }

    /**
     * Update the specified resource in storage.
     *
     * @param ProductBlockRequest $request
     * @param ProductBlock $productBlock
     * @return RedirectResponse|Redirector
     */
    public function update(ProductBlockRequest $request, ProductBlock $productBlock) :RedirectResponse
    {
        // Update image if request has it
        $productBlock->update($request->all());

        // Delete old description
        ProductBlockDescription::where('product_block_id', $productBlock->id)->delete();

        // Insert new description
        $this->saveData( $request, $productBlock->id );

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Handle Save form data
     *
     * @param ProductBlockRequest $request
     * @param int $product_block_id
     * @return void
     */

    private function saveData(ProductBlockRequest $request, int $product_block_id ):void
    {
        $requestData= $request->all();
        $languages= Language::where('status', 1)->cursor();

        foreach ($languages as $lang) {
            $data=[
                'product_block_id'=> $product_block_id,
                'language_id'=> $lang->id,
                'title'=> $requestData['title_'.$lang->local],
                'description'=> $requestData['description_'.$lang->local],
            ];
            ProductBlockDescription::create($data);
        }

    }


    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        if (1 == $id) {
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }

        $productBlock = ProductBlock::withTrashed()->find($id);
        if ($productBlock) {
            if ($productBlock->deleted_at) {
                $productBlock->forceDelete();
            } else {
                $productBlock->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several productBlocks by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ( in_array( 1, $ids) ) {
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }

        if ($request->force) {
            ProductBlock::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            ProductBlock::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $productBlock = ProductBlock::withTrashed()->find($id);

        if ($productBlock){
            $productBlock->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several productBlocks by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $productBlocks = ProductBlock::whereIn('id', $request->ids)->onlyTrashed();
        if ($productBlocks){
            $productBlocks->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
