<?php

namespace App\Http\Controllers\Admin;

use App\Http\Requests\PolicyRequest;
use App\Policy;
use App\PolicyDescription;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Routing\Redirector;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;

class PolicyController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/policies';
    /** View folder */
    private $viewDirectory = 'admin.policies.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }
    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('policies.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the policies.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Policy::orderBy('policies.id', 'ASC')
            ->join('policy_descriptions AS pd', 'policies.id', 'pd.policy_id')
            ->join('languages', 'languages.id', 'pd.language_id')
            ->where('pd.language_id', currentLanguage()->id)
            ->select(['pd.terms_conditions_description', 'pd.privacy_policy_description', 'policies.*']);

        if ($request->date_from) {
            $query->whereDate('policies.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('policies.created_at', '<=', $request->date_to);
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $policies= $query->cursor();

        return view($this->viewDirectory.'grid',get_defined_vars());
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Policy $policy
     * @return View
     */
    public function edit(Policy $policy) :View
    {
        $action= route('policies.update', $policy->id);
        $head = metaFields('policies', 'edit', getCurrentLocale()) ?? __('policies.edit');
        $title = $head;
        $query = PolicyDescription::where('policy_id', $policy->id)
            ->join('languages', 'languages.id', 'policy_descriptions.language_id')
            ->select(['policy_descriptions.*', 'languages.local']);

        $policyDescription= $query->get();

        foreach ($policyDescription as $row) {
            $policy[$row->local]= $row;
        }

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     *
     * @param PolicyRequest $request
     * @param Policy $policy
     * @return RedirectResponse|Redirector
     */
    public function update(PolicyRequest $request, Policy $policy) :RedirectResponse
    {
        $policy->touch();

        // Delete old description
        PolicyDescription::where('policy_id', $policy->id)->delete();

        // Insert new description
        $this->saveData( $request, $policy->id );

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Handle Save form data
     *
     * @param PolicyRequest $request
     * @param int $policy_id
     * @return void
     */

    private function saveData(PolicyRequest $request, int $policy_id ):void
    {
        $requestData= $request->all();

        foreach (languages() as $lang) {
            $data=[
                'policy_id'=> $policy_id,
                'language_id'=> $lang->id,
                'privacy_policy_description'=> $requestData['privacy_policy_description_'.$lang->local],
                'privacy_policy_description_list'=> $requestData['privacy_policy_description_list_'.$lang->local],
                'terms_conditions_description'=> $requestData['terms_conditions_description_'.$lang->local],
            ];
            PolicyDescription::create($data);
        }

    }
}
