<?php

namespace App\Http\Controllers\Admin;

use App\Order;
use App\User;
use Exception;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;

class PendingBasketController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/pendingBaskets';
    /** View folder */
    private $viewDirectory = 'admin.pending_baskets.';

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('pendingBaskets.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index',compact('title'));
    }

    /**
     * return data of the courses.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Order::orderBy('orders.id', 'DESC')
            ->where('orders.status', 'cart_mode')
            ->join('users', 'users.id', 'orders.user_id')
            ->select([
                'orders.*',
                'users.name as user_name',
                'users.phone as user_phone',
                'users.email as user_email',
            ]);

        if ($request->date_from) {
            $query->whereDate('orders.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('orders.created_at', '<=', $request->date_to);
        }

        if ($request->user_name) {
            $query->where('users.name', 'LIKE', '%'.$request->user_name.'%');
        }
        if ($request->user_phone) {
            $query->where('users.phone', 'LIKE', '%'.$request->user_phone.'%');
        }
        if ($request->user_email) {
            $query->where('users.email', 'LIKE', '%'.$request->user_email.'%');
        }


        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $pendingBaskets= $query->paginate(10);

        return view($this->viewDirectory.'grid',compact('pendingBaskets'));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param Order $pendingBasket
     * @return View
     */
    public function edit(Order $pendingBasket)
    {
        $head = __('pendingBaskets.edit');
        if ($pendingBasket->status != 'cart_mode'){
            return  redirect()->route('pendingBaskets.index');
        }

        $orders = Order::where('orders.id', $pendingBasket->id)
            ->join('product_orders as po', 'po.order_id','orders.id')
            ->join('products', 'products.id', 'po.product_id')
            ->join('product_descriptions as pd', 'products.id', 'pd.product_id')
            ->where('pd.language_id', currentLanguage()->id)
            ->select([
                'orders.id',
                'orders.pay_way',
                'orders.created_at',
                'orders.status',
                'po.product_id',
                'po.amount',
                'po.price',
                'po.color',
                'products.images',
                'pd.name as product_name',
            ])->get();

        $user = User::where('id', $pendingBasket->user_id)->first();

        $local = getCurrentLocale();
        $addedValue = array_key_exists('added_value', settings()) ? settings('added_value') : 15;
        $shippingValue = array_key_exists('shipping_value', settings()) ? settings('shipping_value') : 15;


        return view( $this->viewDirectory.'form', compact('orders','pendingBasket', 'user', 'addedValue', 'shippingValue', 'local', 'head'));
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $userProduct= Order::withTrashed()->find($id);
        if ($userProduct) {
            if ($userProduct->deleted_at) {
                $userProduct->forceDelete();
            } else {
                $userProduct->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            Order::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            Order::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $userProduct = Order::withTrashed()->find($id);

        if ($userProduct){
            $userProduct->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $userProduct = Order::whereIn('id', $request->ids)->onlyTrashed();
        if ($userProduct){
            $userProduct->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }


}
