<?php

namespace App\Http\Controllers\Admin;

use App\City;
use App\Http\Controllers\Controller;
use App\Http\Requests\MemberRequest;
use App\Jobs\SendAcceptProviderJob;
use App\Region;
use App\Sector;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

use App\User;

class MemberController extends Controller
{
    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/members';
    /** View folder */
    private $viewDirectory = 'admin.members.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('members.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index',get_defined_vars());
    }

    /**
     * return data of the members.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = User::whereIn('role', ['user', 'provider'])->orderBy('id', 'DESC');

        if ($request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('name', 'LIKE', '%'.$request->name.'%');
        }

        if ($request->phone) {
            $query->where('phone', 'LIKE', '%'.$request->phone.'%');
        }
        if ($request->email) {
            $query->where('email', 'LIKE', '%'.$request->email.'%');
        }

        if ($request->role) {
            $query->where('role', strtolower($request->role) );
        }

        if ($request->user_status) {
            if ($request->user_status == 'active'){
                $query->where('status',1);
            }
            if ($request->user_status == 'non_active'){
                $query->where('status', 0);
            }
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $members= $query->paginate(1000);

        return view($this->viewDirectory.'grid',get_defined_vars());
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('members.store');
        $head = metaFields('members', 'add_new', getCurrentLocale()) ?? __('members.new');
        $title = $head;
        $regions = Region::withDescription();
        $cities = [];
        $sectors = [];
        $local = getCurrentLocale();


        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param MemberRequest $request
     * @return JsonResponse
     */
    public function store(MemberRequest $request) :JsonResponse
    {
        $data = $request->all();
        $data['password'] = bcrypt($request->password);
        $data['contract_agreement'] = $request->contract_agreement ? $request->contract_agreement : 0;
        User::create($data);
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('members.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param User $member
     * @return View
     */
    public function edit(User $member)
    {
        $action= route('members.update', $member->id);
        $head = metaFields('members', 'edit', getCurrentLocale()) ?? __('members.edit');
        $title = $head;
        $regions = Region::withDescription();
        $cities = City::withDescription($member->region_id);
        $sectors = Sector::withDescription($member->city_id);
        $local = getCurrentLocale();
        $preview = strpos($member->commercial_file, '.pdf') > 0 ? '/assets/web/images/image_preview.png' :$member->commercial_file;
//        dd($member->commercial_file);

        return view( $this->viewDirectory.'form', get_defined_vars() );
    }

    /**
     * Update the specified resource in storage.
     *
     * @param MemberRequest $request
     * @param User $member
     * @return JsonResponse
     */
    public function update(MemberRequest $request, User $member) :JsonResponse
    {
        $data = $request->all();
        $data['password'] = $request->password ? bcrypt($request->password) : $member->password;
        $data['contract_agreement'] = $request->contract_agreement ? $request->contract_agreement : 0;
        $member->update($data);
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('members.index')]);

//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }


    public function activeMember(User $member)
    {
        if ($member->status){
            $member->update(['status' => 0]);
            return response()->json(['message' => __('members.de_active_success')]);
        } else {
            $member->update(['status' => 1]);
            if ($member->role == 'provider'){
                try{
                    $data = [
                        'email' => $member->email,
                        'user' => $member->name,
                        'title' => __('members.accept_provider_title'),
                        'message' => __('members.accept_message'),
                    ];
                    SendAcceptProviderJob::dispatch($data, 'web.members.providers.emails.accept_provider'); // himabehiss@gmail.com
                } catch(Exception $exception){}
            }

            return response()->json(['message' => __('members.active_success')]);
        }

    }

    public function getCityAndSector(Request $request)
    {
        if ($request->ajax()){
            $data = [];
            $status = false; // to specify which request returned city or sectors
            if ($request->region_id){
                $status = 'city';
                $data = City::withDescription($request->region_id);
            }
            if ($request->city_id){
                $status = 'sector';
                $data = Sector::withDescription($request->city_id);
            }

            return ['data' => $data, 'status' => $status];
        }
    }



    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $member = User::withTrashed()->find($id);
        if ($member) {
            if ($member->deleted_at) {
                $member->forceDelete();
            } else {
                $member->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            User::onlyTrashed()->whereIn('id', $ids)->where('role', '!=', 'admin')->forceDelete();
        } else {
            User::whereIn('id', $ids)->where('role', '!=', 'admin')->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $member = User::withTrashed()->find($id);

        if ($member){
            $member->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $member = User::whereIn('id', $request->ids)->onlyTrashed();
        if ($member){
            $member->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
