<?php

namespace App\Http\Controllers\Admin;

/** Ajax Request */

use App\Category;
use App\City;
use App\Hall;
use App\Occasion;
use App\Region;
use App\Sector;
use App\Table;
use App\User;
use Exception;
use Illuminate\Support\Carbon;
use Illuminate\View\View;
use App\HallDescription;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;;
use App\Http\Controllers\Controller;
use App\Http\Requests\HallRequest; // for Validation

class HallController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/halls';
    /** View folder */
    private $viewDirectory = 'admin.halls.';

    /**
     * Display a listing of the halls.
     *
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('halls.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index',compact('title'));
    }

    /**
     * return data of the halls.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Hall::latest()
        ->join('hall_descriptions AS hd', 'halls.id', 'hd.hall_id')
        ->join('languages', 'languages.id', 'hd.language_id')
        ->where('local', getCurrentLocale())
        ->select(['hd.name', 'halls.*']);

        if ($request->date_from) {
            $query->whereDate('halls.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('halls.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('hd.name', 'LIKE', '%'.$request->name.'%');
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }
        $halls= $query->paginate(100);

        return view($this->viewDirectory.'grid',compact('halls'));
    }

    /**
     * Show the form for creating a new service.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('halls.store');
        $head = metaFields('halls', 'add_new', getCurrentLocale()) ?? __('halls.new');
        $categories = Category::withDescription();
        $providers = User::where('role', 'provider')->cursor();
        $regions = Region::withDescription();
        $occasions = Occasion::withDescription();
        $tables = Table::withDescription();

        $cities = [];
        $sectors = [];
        $local = getCurrentLocale();
        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created service in storage.
     * @param HallRequest $request
     * @return JsonResponse
     */
    public function store(HallRequest $request) :JsonResponse
    {
        $data = $this->hallData($request);
        $hall = Hall::create($data);
        $this->saveData( $request, $hall->id );
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('halls.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Show the form for editing the specified service.
     *
     * @param  Hall  $hall
     * @return View
     */
    public function edit(Hall $hall) :View
    {
        $action= route('halls.update', $hall->id);
        $head = metaFields('halls', 'edit', getCurrentLocale()) ?? __('halls.edit');

        $query = HallDescription::where('hall_id', $hall->id)
        ->join('languages', 'languages.id', 'hall_descriptions.language_id')
        ->select(['hall_descriptions.*', 'languages.local']);

        $hallDescription= $query->get();

        foreach ($hallDescription as $row) {
            $hall[$row->local]= $row;
        }

        $categories = Category::withDescription();
        $providers = User::where('role', 'provider')->cursor();
        $regions = Region::withDescription();
        $cities = City::withDescription($hall->region_id);
        $sectors = Sector::withDescription($hall->city_id);
        $occasions = Occasion::withDescription();
        $tables = Table::withDescription();

        $local = getCurrentLocale();

        $hall->added_services = json_decode($hall->added_services);

        return view( $this->viewDirectory.'form', get_defined_vars());
    }


    public function activeHall(Hall $hall)
    {
        if ($hall->hall_status){
            $hall->update(['hall_status' => 0]);
            return response()->json(['message' => __('halls.de_active_success')]);
        } else {
            $hall->update(['hall_status' => 1]);
            return response()->json(['message' => __('halls.active_success')]);
        }

    }

    /**
     * Update the specified service in storage.
     * @param HallRequest $request
     * @param Hall $hall
     * @return JsonResponse
     */
    public function update(HallRequest $request, Hall $hall) :JsonResponse
    {
        $data = $this->hallData($request);
        $hall->update($data);
        HallDescription::where('hall_id', $hall->id)->delete();
        $this->saveData( $request, $hall->id );
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('halls.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Handle Save form data
     *
     * @param HallRequest $request
     * @param int $hall_id
     * @return void
     */

     private function saveData(HallRequest $request, int $hall_id ) :void
     {

        $requestData= $request->all();

        foreach (languages() as $lang) {
            $data=[
                'hall_id'=> $hall_id,
                'language_id'=> $lang->id,
                'name'=> $requestData['name_'.$lang->local],
                'description'=> $requestData['description_'.$lang->local],
                'reservation_condition'=> $requestData['reservation_condition_'.$lang->local],
                'features'=> $requestData['features_'.$lang->local],
            ];

            HallDescription::create($data);
        }

     }

     private function hallData($request){
         $data = $request->all();
         $data['added_services'] = [];
         $datesRang = []; // Equal available dates after filter it
         $allUnavailableDates = [];
         $unavailableDates = [];
         $discountDates = [];

         // Generate days between from => to
         $start = \Carbon\Carbon::make($request->dates_rang_from)->addDays(-1);
         $end = Carbon::make($request->dates_rang_to);
         $diff = $end->diff($start)->days;
         if ($diff){
             while($end->diff($start)->days){
                 $start = $start->addDay();
                 $datesRang[] = $start->format('Y-m-d');
             }
         }

         if ($request->unavailable_dates_from){

             foreach ($request->unavailable_dates_from as $fromDateKey => $datesFrom) {
                 $from = Carbon::make($datesFrom)->addDays(-1);
                 $to = Carbon::make($request->unavailable_dates_to[$fromDateKey]);
                 $diff = $to->diff($from)->days;
                 if ($diff){
                     while($to->diff($from)->days){
                         // To filter days and get available dates
//                            if (in_array($from->format('Y-m-d'), $datesRang)){
//                                $key = array_search($from->format('Y-m-d'), $datesRang);
//                                unset($datesRang[$key]);
//                            }
                         $from = $from->addDay();
                         $unavailableDates[$fromDateKey][] = $from->format('Y-m-d');
                         // $allUnavailableDates used to check if it contains in discount dates
                         $allUnavailableDates[] = $from->format('Y-m-d');
                     }
                 }

             }

         }

         if ($request->discount_dates_from){
             foreach ($request->discount_dates_from as $discFromDateKey => $discDatesFrom) {
                 $from = Carbon::make($discDatesFrom)->addDays(-1);
                 $to = Carbon::make($request->discount_dates_to[$discFromDateKey]);
                 $diff = $to->diff($from)->days;
                 if ($diff){
                     while($to->diff($from)->days){
                         $from = $from->addDay();
                         if (in_array($from->format('Y-m-d'), $allUnavailableDates)){
                             return response()->json(['errors' => __('halls.unavailable_dates_cant_be_discount_dates')]);
                         }
                         $discountDates[$discFromDateKey]['dates'][] = $from->format('Y-m-d');
                     }
//                        $discountDates[$discFromDateKey]['discount'] = $request->discount_dates[$discFromDateKey];
                     $discountDates[$discFromDateKey]['discount_percentage'] = $request->discount_percentage[$discFromDateKey] == 'empty' ? 0 : $request->discount_percentage[$discFromDateKey];
                 }

             }
         }

         if ($request->added_services_name){
             foreach ($request->added_services_name as $key => $name) {
                 array_push($data['added_services'], [
                     'id' => ($key+1),
                     'name' => $name,
                     'price' => $request->added_services_price[$key],
                     'description' => $request->added_services_description[$key],
                     'image' => $request->added_services_image[$key],
                 ]);
             }
         }


         $data['added_services'] = json_encode($data['added_services']);
         $data['unavailable_dates'] = $unavailableDates;
         $data['available_dates'] = [
             'dates_rang_from' => $request->dates_rang_from,
             'dates_rang_to' => $request->dates_rang_to,
             'dates' => array_values(array_diff($datesRang, $allUnavailableDates))
         ];
         $data['discount_dates'] = $discountDates;

         return $data;
     }

    public function getCities(Request $request)
    {
        if ($request->ajax()){
            $cities = City::withDescription($request->region_id);
            return ['cities' => $cities];
        }
    }

    public function getCityAndSector(Request $request)
    {
        if ($request->ajax()){
            $data = [];
            $status = false; // to specify which request returned city or sectors
            if ($request->region_id){
                $status = 'city';
                $data = City::withDescription($request->region_id);
            }
            if ($request->city_id){
                $status = 'sector';
                $data = Sector::withDescription($request->city_id);
            }

            return ['data' => $data, 'status' => $status];
        }
    }

    /**
     * Remove the specified service from storage.
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $hall= Hall::withTrashed()->find($id);
        if ($hall) {
//            $hallHasReservations = Reservation::where('hall_id', $id)
//                ->whereIn('re_status', ['confirmed', 'unconfirmed', 'waiting_for_pay'])->count();
//            if ($hallHasReservations){
//                return response()->json(['message'=> __('halls.cant_delete_hall_has_reservation')]);
//            }

            if ($hall->deleted_at) {
                $hall->forceDelete();
            } else {
                $hall->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several halls by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;

        if ($request->force) {
            Hall::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            Hall::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified service from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $hall= Hall::withTrashed()->find($id);
        if ($hall) {
            $hall->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);

    }

    /**
     * Restore several halls by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        Hall::whereIn('id', $ids)->onlyTrashed()->restore();
        return response()->json(['message'=> __('dashboard.saveDone')]);
    }
}
