<?php

namespace App\Http\Controllers\Admin;

use App\Http\Requests\ContractRequest;
use App\Contract;
use App\ContractDescription;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Routing\Redirector;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;

class ContractController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/contracts';
    /** View folder */
    private $viewDirectory = 'admin.contracts.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }
    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('contracts.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the contracts.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Contract::orderBy('contracts.id', 'ASC')
            ->join('contract_descriptions AS cd', 'contracts.id', 'cd.contract_id')
            ->join('languages', 'languages.id', 'cd.language_id')
            ->where('cd.language_id', currentLanguage()->id)
            ->select(['cd.text', 'contracts.*']);

        if ($request->date_from) {
            $query->whereDate('contracts.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('contracts.created_at', '<=', $request->date_to);
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $contracts= $query->cursor();

        return view($this->viewDirectory.'grid',get_defined_vars());
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Contract $contract
     * @return View
     */
    public function edit(Contract $contract) :View
    {
        $action= route('contracts.update', $contract->id);
        $head = metaFields('contracts', 'edit', getCurrentLocale()) ?? __('contracts.edit');
        $title = $head;

        $query = ContractDescription::where('contract_id', $contract->id)
            ->join('languages', 'languages.id', 'contract_descriptions.language_id')
            ->select(['contract_descriptions.*', 'languages.local']);

        $contractDescription= $query->get();

        foreach ($contractDescription as $row) {
            $contract[$row->local]= $row;
        }

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     *
     * @param ContractRequest $request
     * @param Contract $contract
     * @return RedirectResponse|Redirector
     */
    public function update(ContractRequest $request, Contract $contract) :RedirectResponse
    {
        $contract->touch();

        // Delete old description
        ContractDescription::where('contract_id', $contract->id)->delete();

        // Insert new description
        $this->saveData( $request, $contract->id );

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Handle Save form data
     *
     * @param ContractRequest $request
     * @param int $contract_id
     * @return void
     */

    private function saveData(ContractRequest $request, int $contract_id ):void
    {
        $requestData= $request->all();

        foreach (languages() as $lang) {
            $data=[
                'contract_id'=> $contract_id,
                'language_id'=> $lang->id,
                'text'=> $requestData['text_'.$lang->local],
            ];
            ContractDescription::create($data);
        }

    }
}
