<?php

namespace App\Http\Controllers\Admin;

use App\Evaluate;
use App\Hall;
use App\User;
use Exception;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;

class CommentController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/comments';
    /** View folder */
    private $viewDirectory = 'admin.comments.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('comments.head');
        $request= $request->toArray();
        $halls = Hall::whereHas('evaluates')
            ->join('hall_descriptions as hall_desc', 'halls.id', 'hall_desc.hall_id')
            ->where('hall_desc.language_id', currentLanguage()->id)
            ->select(['halls.id', 'hall_desc.name'])->cursor();

        $users = User::whereHas('evaluates')->where('role', 'user')->cursor();

        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the pages.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Evaluate::join('users', 'users.id', 'evaluates.user_id')
        ->join('halls', 'halls.id', 'evaluates.hall_id')
        ->join('hall_descriptions as hall_desc', 'halls.id', 'hall_desc.hall_id')
        ->where('hall_desc.language_id', currentLanguage()->id)
        ->select(['evaluates.*', 'users.name as user_name', 'users.id as user_id', 'halls.id as hall_id' , 'hall_desc.name as hall_name']);


        if ($request->date_from) {
            $query->whereDate('evaluates.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('evaluates.created_at', '<=', $request->date_to);
        }

        if ($request->comment) {
            $query->where('evaluates.comment', 'LIKE', '%'.$request->comment.'%');
        }

        if ($request->user_id) {
            $query->where('users.id', $request->user_id);
        }

        if ($request->hall_id) {
            $query->where('halls.id', $request->hall_id);
        }


        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $comments = $query->paginate(100);

        return view($this->viewDirectory.'grid',get_defined_vars());
    }
    /**
     * Show the form for editing the specified resource.
     *
     * @param Evaluate $comment
     * @return View
     */
    public function edit(Evaluate $comment) :View
    {
        $head = __('comments.comment_details');
        $title = $head;
        $myComment = Evaluate::where('evaluates.id', $comment->id)
            ->join('users', 'users.id', 'evaluates.user_id')
            ->join('halls', 'halls.id', 'evaluates.hall_id')
            ->join('hall_descriptions as hall_desc', 'halls.id', 'hall_desc.hall_id')
            ->where('hall_desc.language_id', currentLanguage()->id)
            ->select(
                ['evaluates.*',
                    'users.name as user_name',
                    'users.id as user_id',
                    'halls.id as hall_id',
                    'hall_desc.name as hall_name'
                ])->first();

        $comment = $myComment;

        return view( $this->viewDirectory.'form', get_defined_vars());
    }


    public function activeComment(Evaluate $comment)
    {
        if ($comment->status){
            $comment->update(['status' => 0]);
            return response()->json(['message' => __('comments.de_active_success')]);
        } else {
            $comment->update(['status' => 1]);
            return response()->json(['message' => __('comments.active_success')]);
        }

    }


    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $comment = Evaluate::withTrashed()->find($id);
        if ($comment) {
            if ($comment->deleted_at) {
                $comment->forceDelete();
            } else {
                $comment->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            Evaluate::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            Evaluate::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $comment = Evaluate::withTrashed()->find($id);

        if ($comment){
            $comment->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $comments = Evaluate::whereIn('id', $request->ids)->onlyTrashed();
        if ($comments){
            $comments->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
