<?php

namespace App\Http\Controllers\Admin;

use App\About;
use App\AboutDescription;
use App\Http\Requests\AboutRequest;
use Exception;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Redirector;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;

class AboutController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/abouts';
    /** View folder */
    private $viewDirectory = 'admin.abouts.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }
    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('abouts.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index',compact('title'));
    }

    /**
     * return data of the abouts.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = About::orderBy('abouts.id', 'DESC')
            ->join('about_descriptions AS aboutDesc', 'abouts.id', 'aboutDesc.about_id')
            ->join('languages', 'languages.id', 'aboutDesc.language_id')
            ->where('aboutDesc.language_id', currentLanguage()->id)
            ->select(['aboutDesc.title', 'abouts.*']);

        if ($request->date_from) {
            $query->whereDate('abouts.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('abouts.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('aboutDesc.title', 'LIKE', '%'.$request->name.'%');
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $abouts= $query->cursor();

        return view($this->viewDirectory.'grid',compact('abouts'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param About $about
     * @return View
     */
    public function edit(About $about) :View
    {
        $action= route('abouts.update', $about->id);
        $head = metaFields('abouts', 'edit', getCurrentLocale()) ?? __('abouts.edit');
        $title = $head;

        $query = AboutDescription::where('about_id', $about->id)
            ->join('languages', 'languages.id', 'about_descriptions.language_id')
            ->select(['about_descriptions.*', 'languages.local']);

        $aboutDescription= $query->get();

        foreach ($aboutDescription as $row) {
            $about[$row->local]= $row;
        }

        $local = getCurrentLocale();

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     *
     * @param AboutRequest $request
     * @param About $about
     * @return RedirectResponse|Redirector
     */
    public function update(AboutRequest $request, About $about) :RedirectResponse
    {
        $about->update($request->all());

        // Delete old description
        AboutDescription::where('about_id', $about->id)->delete();

        // Insert new description
        $this->saveData( $request, $about->id );

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Handle Save form data
     *
     * @param AboutRequest $request
     * @param int $about_id
     * @return void
     */

    private function saveData(AboutRequest $request, int $about_id ):void
    {
        $requestData= $request->all();

        foreach (languages() as $lang) {
            $data=[
                'about_id'=> $about_id,
                'language_id'=> $lang->id,
                'title'=> $requestData['title_'.$lang->local],
                'description'=> $requestData['description_'.$lang->local],
                'about_us_list'=> $requestData['about_us_list_'.$lang->local],
                'why_us_first_title'=> $requestData['why_us_first_title_'.$lang->local],
                'why_us_second_title'=> $requestData['why_us_second_title_'.$lang->local],
                'why_us_third_title'=> $requestData['why_us_third_title_'.$lang->local],
                'why_us_first_description'=> $requestData['why_us_first_description_'.$lang->local],
                'why_us_second_description'=> $requestData['why_us_second_description_'.$lang->local],
                'why_us_third_description'=> $requestData['why_us_third_description_'.$lang->local],
            ];
            AboutDescription::create($data);
        }

    }
}
