<?php

use App\Module;
use App\Service;
use App\User;
use App\Setting;
use App\Language;
use App\Category;
use App\Helpers\ImageHelper;
use App\Newsletter;
use App\VideoCategory;
use App\Providers\AppServiceProvider;
use Illuminate\Support\Carbon;
use Illuminate\Support\LazyCollection;
use Illuminate\Support\Str;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;

if (!function_exists('languages')){

    /**
     * Return list of languages from database
     * @return LazyCollection
     */
    function languages(): LazyCollection
    {
        $languages = Language::where('status', 1)->cursor();
        return $languages;
    }
}

if (!function_exists('currentLanguage')){

    /**
     * Return current language from database
     * @return mixed
     */
    function currentLanguage()
    {
        $languages = Language::where('status', 1)->where('local', LaravelLocalization::getCurrentLocale())->first();
        return $languages;
    }
}

if (!function_exists('getCurrentLocale')){

    /**
     * Return current current locale from database
     * @return string
     */
    function getCurrentLocale(): string
    {
        $current = Language::where('status', 1)->where('local', LaravelLocalization::getCurrentLocale())->first()->local;
        return $current;
    }
}

if (!function_exists('descriptions')){

    /**
     * Return description for model
     * @param $model
     * @param null $local_lang
     * @return mixed
     */
    function descriptions($model, $local_lang = null)
    {
        $description = $model->descriptions->where('language_id', $local_lang ?? currentLanguage()->id)->first();
        return $description ?? null;
    }
}

if (!function_exists('parseExternalVideoLink')){

    /**
     * Parse youtube|vimo|etc link to get video ID, video embed link, video thumbnails
     * @param $link
     * @return array
     */
    function parseExternalVideoLink($link) :array
    {

        /**
         * Links to test
         * https://youtu.be/N6wn8zMRlVE
         * https://www.youtube.com/watch?v=zlnau3yk0oU
         * https://www.youtube.com/watch?v=zlnau3yk0oU&features=adsdsd
         * https://www.youtube.com/embed/OC9zn5BKlC0
         */

        $linkPattern = '/^(https:\/\/(www\.)?(youtu|youtube){1}.([a-z]{1,4}))(\/watch\?v=|\/embed\/|\/){0,1}([a-zA-z0-9]{2,15})(&.)*/';

        $m= preg_match($linkPattern, $link, $linkMatches);

        // dd($m, $linkMatches);
        if (count($linkMatches) && $linkMatches[6]){
            $code = $linkMatches[6];
            $video_url = 'https://www.youtube.com/embed/'.$code;
            $image = "https://img.youtube.com/vi/$code/hqdefault.jpg";
        } else {
            $video_url = $link;
        }

        // $public = asset('/uploads/images/'.time().'.jpg');

        // Download images from remote server
        // $in=    fopen($image, "rb");
        // $out=   fopen($public, "wb");
        // while ($chunk = fread($in,8192))
        // {
        //     fwrite($out, $chunk, 8192);
        // }
        // fclose($in);
        // fclose($out);


        return [
            'code' => $code,
            'url' => $video_url,
            'image' => $image
        ];

    }
}


if (!function_exists('newsletters')){

    /**
     * Return all newsletters email
     * @return LazyCollection
     */
    function newsletters() :LazyCollection
    {
        $newsletters = Newsletter::cursor();
        return $newsletters;
    }
}

if (!function_exists('getUser')){

    /**
     * Return admin by id
     * @param $id
     * @return mixed
     */
    function getUser($id)
    {
        $user = User::find($id);
        return $user;
    }
}
if (!function_exists('settings')){

    /**
     * Return settings
     * @param string|array $key
     * @return array
     */
    function settings($key = null)
    {
        if (is_array($key)){
            $allSettings = Setting::whereIn('key', $key)->cursor()->toArray();
            foreach ($allSettings as $setting) {
                $settings[$setting['key']] = $setting['value'];
            }
        } elseif($key){
            $settings = Setting::where('key', $key)->first()->value;
        } else {
            $allSettings = Setting::cursor()->toArray();
            foreach ($allSettings as $setting) {
                $settings[$setting['key']] = $setting['value'];
            }
        }
        return $settings;
    }

}


if (!function_exists('meta')){

    /**
     * Return meta fields from module table
     * by field name and local language
     * this functions used mostly in blades views
     * to get meta field to each path individually
     * @param $field
     * @param $local
     * @return string|null
     */
    function meta($field, $local): ?string
    {
        return AppServiceProvider::getMeta()['fields'][$local][$field] ?? null;
    }

}

if (!function_exists('metaFields')){

    /**
     * Return meta field to module by module path from modules table
     * most used in blades views
     * @param $modulePath
     * @param null $fieldName
     * @param null $local
     * @return mixed|null
     */
    function metaFields($modulePath, $fieldName = null, $local = null)
    {
        $meta = Module::where('path', $modulePath)->pluck('meta')->toArray();
        $meta = array_first($meta);
        $meta = $fieldName && $local ? $meta['fields'][$local][$fieldName] ?? null : $meta['fields'];
        return $meta ?? null;
    }

}

if (!function_exists('moduleName')){

    /**
     * Return module name to module by module path from modules table
     * most used in blades views
     * @param $modulePath
     * @param null $local
     * @return mixed
     */
    function moduleName($modulePath, $local = null)
    {
        $meta = Module::where('path', $modulePath)->pluck('meta')->toArray();
        $meta = array_first($meta);
        $meta = $local ? $meta['module_name'][$local] : $meta['module_name'];
        return $meta ?? null;
    }

}

//
if (!function_exists('wordLimit')){

    /**
     * Return limit words
     * @param $string
     * @param $limit
     * @return string
     */
    function wordLimit($string, $limit = null): string
    {
        return Str::words(strip_tags($string), $limit, $end = '...');
    }

}
//

if (!function_exists('getImage')){

    /**
     * A short_cut function calls Imagehelper class to
     * Return the url of the image
     *
     * @param $file
     * @return string
     */
    function getImage($file): string
    {
        return ImageHelper::get($file);
    }

}

if (!function_exists('getFile')){

    /**
     * A get URL of the file if it exists, otherwise return null
     * Return the url of the image
     *
     * @param $file
     * @return string
     */
    function getFile($file): ?string
    {
        if (!$file) {
            return null;
        }
        $filePath= public_path($file);
        // dump($filePath , is_file($filePath));
        if (is_file($filePath)) {
            return asset($file);
        } else {
            return null;
        }
    }

}

if (!function_exists('isActive')){

    /**
     * Return module name to module by module path from modules table
     * most used in blades views
     * @param $url
     * @param $class
     * @return mixed
     */
    function isActive($url, $class = 'active')
    {
        $currentUrl = url()->current();
        if ($currentUrl == $url){
            return $class;
        }
        return '';
    }



}

if (!function_exists('stripText')){

    /**
     * Return module name to module by module path from modules table
     * most used in blades views
     * @param $text
     * @param int $limit
     * @param bool $stripSpace
     * @return mixed
     */
    function stripText($text, int $limit = null, $stripSpace = false)
    {
        $description = strip_tags(html_entity_decode($text));
        $description = preg_replace('/\s\s+/', ' ', $description);
        if ($limit){
            $description = \Illuminate\Support\Str::limit($description, $limit);
        }
        if ($stripSpace){
            $description = str_replace(' ', '', $description);
        }
        return $description ?? '';
    }

}



if (!function_exists('getAbout')){

    function getAbout()
    {
        return \App\About::getAbout() ?? [];
    }

}

if (!function_exists('getServices')){

    function getServices($services)
    {
        $names = [];
        $total = 0;
        if (count($services)){
            foreach ($services as $service) {
                foreach ($service as $oneService) {
                    $names[$oneService['id']] = $oneService['name'];
                    $total += $oneService['price']*$oneService['quantity'];
                }
            }
        }

        return ['names' => $names, 'total' => $total];
    }

}

if (!function_exists('getOccasions')){

    function getOccasions($ids)
    {
        $occasions = \App\Occasion::withDescription($ids) ?? [];
        return $occasions ?? [];
    }

}


if (!function_exists('dateInfo')){

    function dateInfo($date, $params = [])
    {
        $dateInfo = Carbon::make($date);
        $params['day_name'] = array_key_exists('day_name', $params) ? $dateInfo->getTranslatedDayName() : null;
        $params['format'] = array_key_exists('format', $params) ? $dateInfo->format($params['format']) : null;
        return $params['day_name'] ?? $params['format'] ;
    }

}


//if (!function_exists('getAgentPhotos')){
//
//    function getAgentPhotos($photo)
//    {
//        $useragent = request()->header('User-Agent');
//        if(preg_match('/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows (ce|phone)|xda|xiino/i',$useragent)||preg_match('/1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i',substr($useragent,0,4))){
//            $path = trim($photo, '/');
//            $path = explode('/', $path);
//            $photo = last($path);
//            array_pop($path);
//            $path = implode('/', $path);
//            $finalImagePath = $path.'/thumbs/'.$photo;
//            return asset($finalImagePath);
//        }
//        return asset($photo);
//    }
//
//}


if (!function_exists('getPhotoThumb')){

    function getPhotoThumb($photo)
    {
        if ($photo){
            $path = trim($photo, '/');
            $path = explode('/', $path);
            $photo = last($path);
            array_pop($path);
            $path = implode('/', $path);
            $finalImagePath = $path.'/thumbs/'.$photo;
            return asset($finalImagePath);
        }

        return '';

    }

}







