<?php

namespace App\Console\Commands;

use App\Hall;
use App\Helpers\ImageHelper;
use App\Http\Controllers\Web\SendNotification;
use App\Reservation;
use App\UsersNotification;
use Illuminate\Console\Command;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;

class AutoCancelReservation extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'reservation:cancel';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Command description';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        Log::info('Task schedule run successfully from Auto cancel reservation');
        $date = Carbon::now()->addHours(-1);
        $reservations = Reservation::whereIn('reservations.re_status', ['unconfirmed', 'waiting_for_pay'])
            ->where('reservations.canceled_from', '=', null)
            ->where('reservations.created_at', '<=', $date)
            ->join('halls', 'halls.id', '=', 'reservations.hall_id')
            ->join('hall_descriptions as hall_desc', 'halls.id', '=', 'hall_desc.hall_id')
            ->where('hall_desc.language_id', currentLanguage()->id)
            ->select([
                'reservations.id',
                'reservations.user_id',
                'reservations.provider_id',
                'reservations.re_status',
                'reservations.re_no',
                'reservations.re_date_from',
                'reservations.person_count',
                'reservations.total',
                'reservations.created_at',
                'reservations.canceled_from',
                'halls.id as hall_id',
                'hall_desc.name as hall_name'])->get();

        foreach ($reservations as $reservation) {

            $user = $reservation->user;
            $provider = $reservation->provider;
            $hall = Hall::withDescription(null, $reservation->hall_id);
            $title = __('reservations.reservation_canceled_title') . "($hall->hall_name)";

            /**
             * The request was not accepted by the service provider
             */
            if ($reservation->re_status == 'unconfirmed'){
                $reservation->update(['re_status' => 'canceled', 'canceled_from' => 'provider']);
                $message = __('reservations.service_provider_not_accept_request');
            }

            /**
             * The request has been accepted by the service provider
             * and the user has not made the payment
             */
            if ($reservation->re_status == 'waiting_for_pay'){
                $reservation->update(['re_status' => 'canceled', 'canceled_from' => 'user']);
                $message = __('reservations.user_not_make_paid_message');
            }

            $messageForUser = str_replace_array('@@', [$user->name, $hall->hall_name, $provider->name], $message);
            $messageForProvider = str_replace_array('@@', [$provider->name, $hall->hall_name, $user->name], $message);

            $userNotification = UsersNotification::create([
                'title' => $title,
                'message' => $messageForUser,
                'user_id' => $user->id,
                'notification_type' => 'cancel',
                'page_link' => route('reservation.reservationDetails',$reservation->id),
            ]);
            $providerNotification = UsersNotification::create([
                'title' => $title,
                'message' => $messageForProvider,
                'user_id' => $provider->id,
                'notification_type' => 'cancel',
                'page_link' => route('reservation.reservationDetails',$reservation->id),

            ]);

            $icon = asset('assets/web/images/icons/profile/noti-canceled.svg');
            $now = \Carbon\Carbon::now();
            $notificationTime = Carbon::make($providerNotification->created_at);
            $notificationTime = $now->diffInMinutes($notificationTime);
            $data = [
                'reservation_done' => 'done',
                'reservation_route' => route('reservation.reservationDetails',$reservation->id),
                'reservation_image' => asset(is_string($hall->images) ? json_decode($hall->images, true)[0] : $hall->images[0]),
                'hall_name' => $hall->hall_name,
                're_no' => "$reservation->re_no",
                're_date_from' => date('d/m/Y', strtotime($reservation->re_date_from)),
                'person_count' => "$reservation->person_count",
                'total' => "$reservation->total",
                'user_photo' => asset($user->photo ?? 'assets/web/images/avatar.png'),
                'notification_time' =>$notificationTime . __('halls.minutes') . __('halls.ago'),
                'notification_date' => date('M Y d', strtotime($providerNotification->created_at)),
                'notification_icon' => $icon,
                'notification_class' => 'canceledItem',
                'reservation_container_id' => '#pills-canceled',
                'badge' => 'badge_canceled',
                'badge_title' => __('reservations.canceled'),
                'change_reservation_status' => 'yes',
            ];
            $userData = array_merge([
                'name' => $user->name,
                'notification_title' => $title,
                'notification_message' => $messageForUser,
                'notification_id' => $userNotification->id
            ], $data);
            $providerData = array_merge([
                'name' => $provider->name,
                'notification_title' => $title,
                'notification_message' => $messageForProvider,
                'notification_id' => $providerNotification->id
            ], $data);


            if (is_array($user->device_token) && isset($user->device_token['token'])){
                $notificationForUser = new SendNotification($title, $messageForUser, $icon, $userData, route('user.profile'));
                $user->notify($notificationForUser);
            }
            if (is_array($user->device_token) && isset($user->device_token['token'])){
                $notificationForProvider = new SendNotification($title, $messageForProvider, $icon, $providerData, route('provider.profile'));
                $provider->notify($notificationForProvider);
            }
        }

    }
}
